/* $Id: $
 *
 * Copyright (c) 1997, 1998, 1999 Systemics Ltd on behalf of
 * the Cryptix Development Team.  All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General License along with this library; if
 * not, you can download a copy from <http://www.cryptix.org/>.
 */

/**
 * A superclass of all asn1.Parser Visitors. Implements a simple iteration
 * through all the children nodes of a non-termianl node for an already
 * generated/parsed sub-tree.<p>
 *
 * This class incorporates design ideas implemented in Pekka Nikander's
 * (main architect & principal programmer;
 * <mailto:Pekka.Nikander@nixu.fi>) JaSCA (Java SNMP Control Applet) public
 * domain software.
 *
 * <b>Copyright</b> &copy;1997, 1998, 1999
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: 1.1 $</b>
 * @author  Raif S. Naffah
 */

package cryptix.asn1.encoding;

import cryptix.asn1.lang.*;

import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;


public class BaseCoder // implicit no-argument constructor
implements CoderOperations // defined by us
{
// Debugging methods and fields
//...........................................................................

    private static final String NAME = "BaseCoder";

    private static final boolean IN = true, OUT = false;
    private static final boolean DEBUG =
        PackageProperties.GLOBAL_DEBUG;
    private static final int debuglevel =
        DEBUG ? PackageProperties.getLevel(NAME) : 0;
    private static final PrintWriter err =
        DEBUG ? PackageProperties.getOutput() : null;
    private static final boolean TRACE =
        PackageProperties.isTraceable(NAME);

    private static void debug(String s) {
        err.println(">>> "+NAME+": "+s);
    }

    private static void trace(boolean in, String s) {
        if (TRACE) err.println((in ? "==> " : "<== ")+NAME+"."+s);
    }

    private static void trace(String s) {
        if (TRACE) err.println("<=> "+NAME+"."+s);
    }


// Constants and vars
//...........................................................................

    // name of the encoding home property key
    private static final String ENCODING_HOME = "asn.1.encoding.home";

    protected InputStream in;
    protected OutputStream out;
    private int state = CoderOperations.UNINITIALIZED; // state of this coder


// Factory method
//...........................................................................

    /**
     * Instantiates an ASN.1 coder to use for concrete processing
     * of the input/output.<p>
     *
     * Effectively, EncodingFactory objects implement the encoding
     * format for an ASN.1 grammar.
     *
     * @param anEncoding Name of the encoding. If the name is not
     *        fully qualified, then look for a class with same name
     *        in package defined by the package-specific property
     *        with the key "asn.1.encoding.home".
     */
    public static CoderOperations getInstance(String anEncoding) {
        int i = anEncoding.lastIndexOf('.');
        if (i == -1) {
            String home = PackageProperties.getProperty(ENCODING_HOME);
            anEncoding = home+"."+anEncoding;
        }
        CoderOperations result = null;
        try {
            result = (CoderOperations) Class.forName(anEncoding).newInstance();
        } catch (Throwable t) {
            debug("Unable to instantiate "+anEncoding+" coder");
            debug(t.toString());
            debug(t.getMessage());
            t.printStackTrace(err);
        }

        return result;
    }


// Accessors
//...........................................................................

    public int getState() {
        return state;
    }


// ParserVisitor default implementations
//...........................................................................

    public Object visit(SimpleNode x, Object data)
    throws IOException {
        String msg = "Don't know how to visit "+x;
        if (DEBUG && debuglevel > 8) debug(msg);
        throw new RuntimeException(msg);
    }

    public Object visit(ASNSpecification node, Object data)
    throws IOException {
        return node.childrenAccept(this, data);
    }

    public Object visit(ASNTypeAlias node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNType node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNBoolean node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNInteger node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNBitString node, Object data)
    throws IOException {
        return node.childrenAccept(this, data);
    }

    public Object visit(ASNOctetString node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNNull node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNObjectIdentifier node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNSequence node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNSequenceOf node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNSet node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNSetOf node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNTaggedType node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNAny node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNPrintableString node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }

    public Object visit(ASNTime node, Object data)
    throws IOException {
        return visitInternal(node, data);
    }


// utility methods
//...........................................................................

    protected Object visitInternal(SimpleNode node, Object data)
    throws IOException {
        if (state == CoderOperations.ENCODING)
            switch (node.getID()) {
            case ParserTreeConstants.JJTTYPE:
                encode((ASNType) node, out);
                break;
            case ParserTreeConstants.JJTTYPEALIAS:
                encode((ASNTypeAlias) node, out);
                break;
            case ParserTreeConstants.JJTBOOLEAN:
                encode((ASNBoolean) node, out);
                break;
            case ParserTreeConstants.JJTINTEGER:
                encode((ASNInteger) node, out);
                break;
            case ParserTreeConstants.JJTBITSTRING:
                encode((ASNBitString) node, out);
                break;
            case ParserTreeConstants.JJTOCTETSTRING:
                encode((ASNOctetString) node, out);
                break;
            case ParserTreeConstants.JJTNULL:
                encode((ASNNull) node, out);
                break;
            case ParserTreeConstants.JJTOBJECTIDENTIFIER:
                encode((ASNObjectIdentifier) node, out);
                break;
            case ParserTreeConstants.JJTSEQUENCE:
                encode((ASNSequence) node, out);
                break;
            case ParserTreeConstants.JJTSEQUENCEOF:
                encode((ASNSequenceOf) node, out);
                break;
            case ParserTreeConstants.JJTSET:
                encode((ASNSet) node, out);
                break;
            case ParserTreeConstants.JJTSETOF:
                encode((ASNSetOf) node, out);
                break;
            case ParserTreeConstants.JJTTAGGEDTYPE:
                encode((ASNTaggedType) node, out);
                break;
            case ParserTreeConstants.JJTANY:
                encode((ASNAny) node, out);
                break;
            case ParserTreeConstants.JJTPRINTABLESTRING:
                encode((ASNPrintableString) node, out);
                break;
            case ParserTreeConstants.JJTTIME:
                encode((ASNTime) node, out);
                break;
            default:
                String msg = "Don't know how to encode "+node;
                if (DEBUG && debuglevel > 8)
                    debug(msg);
                throw new RuntimeException(msg);
            }
        else if (state == CoderOperations.DECODING)
            switch (node.getID()) {
            case ParserTreeConstants.JJTTYPE:
                decode((ASNType) node, in);
                break;
            case ParserTreeConstants.JJTTYPEALIAS:
                decode((ASNTypeAlias) node, in);
                break;
            case ParserTreeConstants.JJTBOOLEAN:
                decode((ASNBoolean) node, in);
                break;
            case ParserTreeConstants.JJTINTEGER:
                decode((ASNInteger) node, in);
                break;
            case ParserTreeConstants.JJTBITSTRING:
                decode((ASNBitString) node, in);
                break;
            case ParserTreeConstants.JJTOCTETSTRING:
                decode((ASNOctetString) node, in);
                break;
            case ParserTreeConstants.JJTNULL:
                decode((ASNNull) node, in);
                break;
            case ParserTreeConstants.JJTOBJECTIDENTIFIER:
                decode((ASNObjectIdentifier) node, in);
                break;
            case ParserTreeConstants.JJTSEQUENCE:
                decode((ASNSequence) node, in);
                break;
            case ParserTreeConstants.JJTSEQUENCEOF:
                decode((ASNSequenceOf) node, in);
                break;
            case ParserTreeConstants.JJTSET:
                decode((ASNSet) node, in);
                break;
            case ParserTreeConstants.JJTSETOF:
                decode((ASNSetOf) node, in);
                break;
            case ParserTreeConstants.JJTTAGGEDTYPE:
                decode((ASNTaggedType) node, in);
                break;
            case ParserTreeConstants.JJTANY:
                decode((ASNAny) node, in);
                break;
            case ParserTreeConstants.JJTPRINTABLESTRING:
                decode((ASNPrintableString) node, in);
                break;
            case ParserTreeConstants.JJTTIME:
                decode((ASNTime) node, in);
                break;
            default:
                String msg = "Don't know how to decode "+node;
                if (DEBUG && debuglevel > 8)
                    debug(msg);
                throw new RuntimeException(msg);
            }
        else
            throw new IllegalStateException();

        return node.getValue();
    }


// EncodingOperations interface methods
// Default (null) implementations follow
//...........................................................................

    public void init(OutputStream os) {
        out = os;
        state = CoderOperations.ENCODING;
    }

    public void init(InputStream is) {
        in = is;
        state = CoderOperations.DECODING;
    }


// Encoding methods
//...........................................................................

    public void encode(ASNType obj, OutputStream out) throws IOException {}
    public void encode(ASNTypeAlias obj, OutputStream out) throws IOException {}
    public void encode(ASNBoolean obj, OutputStream out) throws IOException {}
    public void encode(ASNInteger obj, OutputStream out) throws IOException {}
    public void encode(ASNBitString obj, OutputStream out) throws IOException {}
    public void encode(ASNOctetString obj, OutputStream out) throws IOException {}
    public void encode(ASNNull obj, OutputStream out) throws IOException {}
    public void encode(ASNObjectIdentifier obj, OutputStream out) throws IOException {}
    public void encode(ASNSequence obj, OutputStream out) throws IOException {}
    public void encode(ASNSequenceOf obj, OutputStream out) throws IOException {}
    public void encode(ASNSet obj, OutputStream out) throws IOException {}
    public void encode(ASNSetOf obj, OutputStream out) throws IOException {}
    public void encode(ASNTaggedType obj, OutputStream out) throws IOException {}
    public void encode(ASNAny obj, OutputStream out) throws IOException {}
    public void encode(ASNPrintableString obj, OutputStream out) throws IOException {}
    public void encode(ASNTime obj, OutputStream out) throws IOException {}


// Decoding methods
//...........................................................................

    public void decode(ASNBoolean obj, InputStream in) throws IOException {}
    public void decode(ASNInteger obj, InputStream in) throws IOException {}
    public void decode(ASNBitString obj, InputStream in) throws IOException {}
    public void decode(ASNOctetString obj, InputStream in) throws IOException {}
    public void decode(ASNNull obj, InputStream in) throws IOException {}
    public void decode(ASNObjectIdentifier obj, InputStream in) throws IOException {}
    public void decode(ASNSequence obj, InputStream in) throws IOException {}
    public void decode(ASNSequenceOf obj, InputStream in) throws IOException {}
    public void decode(ASNSet obj, InputStream in) throws IOException {}
    public void decode(ASNSetOf obj, InputStream in) throws IOException {}
    public void decode(ASNTaggedType obj, InputStream in) throws IOException {}
    public void decode(ASNAny obj, InputStream in) throws IOException {}
    public void decode(ASNPrintableString obj, InputStream in) throws IOException {}
    public void decode(ASNTime obj, InputStream in) throws IOException {}

    public void decode(ASNType obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        boolean optional = obj.isOptional();
        String refType = obj.getName();
        SimpleNode x;
        Object value;
        if (refType != null) {
            x = (SimpleNode) obj.getParser().resolve(refType);
        } else
            x = (SimpleNode) obj.getChild(0);

        x.setOptional(optional);
        value = visitInternal(x, null);
        obj.setValue(value);

        if (DEBUG) trace(OUT, "decode(Type) --> "+obj);
    }

    public void decode(ASNTypeAlias obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        SimpleNode x = (SimpleNode) obj.getParser().resolve(obj.getName());
        Object value = visitInternal(x, null);
        obj.setValue(value);

        if (DEBUG) trace(OUT, "decode(TypeAlias) --> "+obj);
    }
}