/* $Id: $
 *
 * Copyright (c) 1997, 1998, 1999 Systemics Ltd on behalf of
 * the Cryptix Development Team.  All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General License along with this library; if
 * not, you can download a copy from <http://www.cryptix.org/>.
 */

package cryptix.asn1.encoding;

import cryptix.asn1.lang.*;
import cryptix.util.core.Hex;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PushbackInputStream;
import java.io.PrintWriter;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.Vector;

/**
 * A concrete factory class to handle ASN.1 DER encoding.<p>
 *
 * Portions of this class's code are ported from the "Unsupported
 * Release of the Certificate Library" in C++ from Sun Microsystems'
 * Internet Commerce Group (see copyright notice later).
 *
 * <b>Copyright</b> &copy;1997, 1998, 1999
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: 1.3 $</b>
 * @author  Raif S. Naffah
 * @author  Eric Rescorla
 */
public class DER // implicit no-argument package-private constructor
extends BaseCoder
{
// Debugging methods and fields
//...........................................................................

    private static final String NAME = "DER";

    private static final boolean IN = true, OUT = false;
    private static final boolean DEBUG =
        PackageProperties.GLOBAL_DEBUG;
    private static final int debuglevel =
        DEBUG ? PackageProperties.getLevel(NAME) : 0;
    private static final PrintWriter err =
        DEBUG ? PackageProperties.getOutput() : null;
    private static final boolean TRACE =
        PackageProperties.isTraceable(NAME);

    private static void debug(String s) {
        err.println(">>> "+NAME+": "+s);
    }

    private static void trace(boolean in, String s) {
        if (TRACE) err.println((in?"==> ":"<== ")+NAME+"."+s);
    }

    private static void trace(String s) {
        if (TRACE) err.println("<=> "+NAME+"."+s);
    }


// Constants and vars
//...........................................................................


// Encoding methods
//...........................................................................

    /** Encodes a BOOLEAN. */
    public void encode(ASNBoolean obj, OutputStream out)
    throws IOException {
        boolean v = ((Boolean) obj.getValue()).booleanValue();
        out.write(Tag.BOOLEAN); // write the tag
        encodeLength(1, out); // write the length
        out.write(v ? 0x01 : 0x00); // and finally the object's data
    }

    /** Encodes an OCTET STRING. */
    public void encode(ASNOctetString obj, OutputStream out)
    throws IOException {
        byte[] v = (byte[]) obj.getValue();
        out.write(Tag.OCTET_STRING); // write the tag
        encodeLength(v.length, out); // write the length
        out.write(v); // and finally the object's data
    }

    /** Encodes a NULL. */
    public void encode(ASNNull obj, OutputStream out)
    throws IOException {
        out.write(Tag.NULL); // write the tag
        encodeLength(0, out); // write the length
    }

    /** Encodes an OID. OIDs are represented as dotted strings. */
    public void encode(ASNObjectIdentifier obj, OutputStream out)
    throws IOException {
        // convert oid components substrings into ints
        String v = (String) obj.getValue();
        StringTokenizer st = new StringTokenizer(v, ".");
        int[] component = new int[st.countTokens()];
        for (int i = 0; i < component.length; i++)
            component[i] = Integer.parseInt(st.nextToken());
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.write(component[0] * 40 + component[1]); // first 2 are special
        int j, c;
        byte[] ab;
        for (int i = 2; i < component.length; i++) {
            c = component[i];
            ab = new byte[4];
            for (j = 0; j < 4; j++) {
                ab[j] = (byte)(c & 0x7F);
                c >>>= 7;
                if (c == 0)
                    break;
            }
            for (; j > 0; j--)
                baos.write(ab[j] | 0x80);
            baos.write(ab[0]);
        }
        out.write(Tag.OBJECT_IDENTIFIER); // write the tag
        encodeLength(baos.size(), out); // write the length
        baos.writeTo(out); // and finally the object's data
    }

    /** Encodes a SEQUENCE. */
    public void encode(ASNSequence obj, OutputStream out)
    throws IOException {
        Object[] v = (Object[]) obj.getValue();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        //
        //
        //
        out.write(Tag.SEQUENCE); // write the tag
        encodeLength(baos.size(), out); // write the length
        baos.writeTo(out); // and finally the object's data
    }

    /** Encodes an ASN.1 Tagged type. */
    public void encode(ASNTaggedType obj, OutputStream out)
    throws IOException {
        //
        // to do
        //
    }

    /** Encodes a UTCTime. */
    public void encode(ASNTime obj, OutputStream out)
    throws IOException {
        //
        // to do
        //
    }


// Decoding methods
//...........................................................................

    /** Decodes a BOOLEAN --internally represented as a Boolean. */
    public void decode(ASNBoolean obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.BOOLEAN, in)) {
            String msg = "Not a BOOLEAN";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            if (length != 1)
                throw new IOException("Incorrect BOOLEAN length: "+length);
            int v = in.read();
            if (v == -1)
                throw new java.io.EOFException();
            obj.setValue(new Boolean(v != 0));
        }

        if (DEBUG) trace(OUT, "decode(BOOLEAN) --> "+obj);
    }

    /** Decodes an INTEGER -- internally represented as a BigInteger. */
    public void decode(ASNInteger obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.INTEGER, in)) {
            String msg = "Not an INTEGER";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            obj.setValue(new BigInteger(1, buffer));
        }

        if (DEBUG) trace(OUT, "decode(INTEGER) --> "+obj);
    }

    /** Decodes a BIT STRING --internally represented as a byte[]. */
    public void decode(ASNBitString obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.BIT_STRING, in)) {
            String msg = "Not a BIT STRING";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            obj.setValue(buffer);
            if (DEBUG && debuglevel > 8) debug("OUT: "+Hex.toString(buffer));
        }

        if (DEBUG) trace(OUT, "decode(BIT STRING)");
    }

    /** Decodes an OCTET STRING --internally represented as a byte[]. */
    public void decode(ASNOctetString obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.OCTET_STRING, in)) {
            String msg = "Not an OCTET STRING";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            obj.setValue(buffer);
            if (DEBUG && debuglevel > 8) debug("OUT: "+Hex.toString(buffer));
        }

        if (DEBUG) trace(OUT, "decode(OCTET STRING)");
    }

    /** Decodes a NULL --internally represented as a null. */
    public void decode(ASNNull obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.NULL, in)) {
            String msg = "Not a NULL";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            if (length != 0)
                throw new IOException("Incorrect NULL length: "+length);
        }

        if (DEBUG) trace(OUT, "decode(NULL)");
    }

    /** Decodes an OID --internally represented as a dot-separated String. */
    public void decode(ASNObjectIdentifier obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.OBJECT_IDENTIFIER, in)) {
            String msg = "Not an OBJECT-IDENTIFIER";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            StringBuffer buffer = new StringBuffer();
            if (--length >= 0) { // first byte is special
                int b      = in.read() & 0xFF;
                int first  = (b < 40 ? 0 : (b < 80 ? 1 : 2));
                int second = (b - first * 40);
                buffer.append(first).append(".").append(second);
            }
            while (length > 0) { // handle the rest
                buffer.append(".");
                int sid = 0; // subid
                int b;
                do {
                    b = in.read() & 0xFF;
                    sid = sid << 7 | (b & 0x7F);
                }
                while (--length > 0 && (b & 0x80) == 0x80);
                buffer.append(sid);
            }
            obj.setValue(new String(buffer));
        }

        if (DEBUG) trace(OUT, "decode(OID) --> "+obj);
    }

    /** Decodes a SEQUENCE --internally represented as an array. */
    public void decode(ASNSequence obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "SEQ: decode("+obj+")");

        decodeSequence(obj, in);

        if (DEBUG) trace(OUT, "decode(SEQUENCE)");
    }

    /** Decodes a SEQUENCE OF --internally represented as an array. */
    public void decode(ASNSequenceOf obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        decodeSequence(obj, in);

        if (DEBUG) trace(OUT, "decode(SEQUENCE OF)");
    }

    /** Decodes a SET --internally represented as an array. */
    public void decode(ASNSet obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        decodeSet(obj, in);

        if (DEBUG) trace(OUT, "decode(SET)");
    }

    /** Decodes a SET OF --internally represented as an array. */
    public void decode(ASNSetOf obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        decodeSet(obj, in);

        if (DEBUG) trace(OUT, "decode(SET OF)");
    }

    /** Decodes a Tagged Type -- rewritten by EKR*/
    public void decode(ASNTaggedType obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        Tag tag = obj.getTag();
        // First, we check to see if the tag on the wire is correct
        in.mark(10);
        if (isExpectedTag(tag, in))
            if (tag.isExplicit())
                decodeExplicitTaggedType(obj, in);
            else {
                in.reset();
                decodeImplicitTaggedType(obj, in);
            }
        else {
            String msg = "Failed to read a non-optional element";
            if (DEBUG && debuglevel > 0) debug(msg);
            // is there a default?
            Object def = obj.getDefaultValue();
            if (def == null && !obj.isOptional())
                throw new IOException(msg);
            if (def != null)
                obj.setValue(def);

            in.reset(); // reset so that the next tag can have it's shot
        }

        if (DEBUG) trace(OUT, "decode(TAGGED TYPE)");
    }

    /** Decodes an ANY, represented as byte[] -- EKR*/
    public void decode(ASNAny obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        byte[] buffer = getTLV(in);
        obj.setValue(buffer);

        if (DEBUG && debuglevel > 8) debug("OUT: "+Hex.toString(buffer));
        if (DEBUG) trace(OUT, "decode(ANY)");
    }

    /** Decodes a PrintableString --internally represented as a String. */
    public void decode(ASNPrintableString obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.PRINT_STRING, in)) {
            String msg = "Not a PrintableString or IA5String";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            obj.setValue(new String(buffer));
        }

        if (DEBUG) trace(OUT, "decode(PrintableString) --> "+obj);
    }

    /** Decodes a UTCTime --internally represented as a Date. */
    public void decode(ASNTime obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "decode("+obj+")");

        if (!isExpectedTag(Tag.UTC_TIME, in)) {
            String msg = "Not a UTC_TIME";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);

            Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
            int YY = (buffer[0]-'0')*10 + (buffer[1]-'0');
            int MM = (buffer[2]-'0')*10 + (buffer[3]-'0') - 1;
            int DD = (buffer[4]-'0')*10 + (buffer[5]-'0');
            int hh = (buffer[6]-'0')*10 + (buffer[7]-'0');
            int mm = (buffer[8]-'0')*10 + (buffer[9]-'0');
            int ss = 0;

            YY += YY <= 50 ? 2000 : 1900; // fails for 2051 and later
            if (buffer[10] != 'Z') {
                ss = (buffer[10]-'0')*10 + (buffer[11]-'0');
                if (buffer[12] != 'Z')
                    throw new IOException("Bad date format");
            }

            cal.set(YY, MM, DD, hh, mm, ss);
            obj.setValue(cal.getTime());
        }

        if (DEBUG) trace(OUT, "decode(UTCTime) --> "+obj);
    }


// own/utility methods
//...........................................................................

    /** Decodes the tag element of a DER data triplet. */
    private boolean isExpectedTag(int expectedTag, InputStream in)
    throws IOException {
        Tag result = Tag.getExpectedTag(expectedTag, in);
        return result != null;
    }

    /** Decodes the tag element of a DER data triplet. */
    private boolean isExpectedTag(Tag tag, InputStream in)
    throws IOException {
        Tag result = Tag.getExpectedTag(tag, in);
        return result != null;
    }

    /** Decodes the length element of a DER data triplet. */
    private static int decodeLength(InputStream in)
    throws IOException {
        int result;
        int limit = in.read();
        if ((limit & 0x80) == 0)
            result = limit;
        else {
            limit &= 0x7F;
//            if (limit <= 0)
//                throw new IOException("DER indefinite form not supported");
            if (limit > 4)
                throw new IOException("ASN.1 DER object too large");
            result = 0;
            while (limit-- > 0)
                result = (result << 8) | (in.read() & 0xFF);
        }

        if (DEBUG && debuglevel > 7) debug("Element length = "+result);
        return result;
    }

    /** Decodes a SEQUENCE --internally represented as a Vector of Objects. */
    private void decodeSequence(SimpleNode obj, InputStream in)
    throws IOException {
        if (!(isExpectedTag(Tag.SEQUENCE, in) ||
              isExpectedTag(Tag.SEQUENCE_OF, in))) {
            String msg = "Not a SEQUENCE [OF]";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            // now parse the contents of this SEQUENCE [OF] into its components
            CoderOperations ber = BaseCoder.getInstance("DER");
            ByteArrayInputStream in2 = new ByteArrayInputStream(buffer);
            ber.init(in2);
            Vector value = new Vector();
            while (in2.available() != 0) {
                Object element = obj.childrenAccept(ber, null);
                value.addElement(element);
            }

            obj.setValue(value);
        }
    }

    /** Decodes a SET --internally represented as a Vector of Objects. */
    private void decodeSet(SimpleNode obj, InputStream in)
    throws IOException {
        if (! (isExpectedTag(Tag.SET, in) ||
               isExpectedTag(Tag.SET_OF, in))) {
            String msg = "Not a SET [OF]";
            if (DEBUG && debuglevel > 8) debug(msg);
            if (!obj.isOptional())
                throw new IOException(msg);
        } else {
            int length = decodeLength(in);
            byte[] buffer = new byte[length];
            int actualLength = in.read(buffer);
            if (actualLength == -1)
                throw new java.io.EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
            // now parse the contents of this SET [OF] into its components
            CoderOperations ber = BaseCoder.getInstance("DER");
            ByteArrayInputStream in2 = new ByteArrayInputStream(buffer);
            ber.init(in2);
            Vector value = new Vector();
            while (in2.available() != 0) {
                Object element = obj.childrenAccept(ber, null);
                value.addElement(element);
            }

            obj.setValue(value);
        }
    }

    /** Decodes an EXPLICIT Tagged Type.*/
    public void decodeExplicitTaggedType(ASNTaggedType obj, InputStream in)
    throws IOException {
        // EKR removed the tag check because it's now performed
        // in decode(ASNTaggedType,InputStream)
        if (DEBUG) trace(IN, "...decodeExplicitTaggedType("+obj+")");

        int length=decodeLength(in);

        byte[] buffer=new byte[length];
        int actualLength = in.read(buffer);
        if (actualLength == -1)
            throw new EOFException();
        if (actualLength != length)
            throw new IOException("Length ("+length+") mismatch: "+actualLength);
        // now parse the contents of this constructed Tagged element
        // into its components
        CoderOperations ber = BaseCoder.getInstance("DER");
        ber.init(new ByteArrayInputStream(buffer));

        SimpleNode x = (SimpleNode) obj.getParser().resolve(obj.getName());
        Object value = x.childrenAccept(ber, null);
        String msg;
/*(	Object oldValue = obj.getValue();
	if (oldValue != null) {
	  msg = "********************* Old value: "+oldValue;
	  if (DEBUG && debuglevel > 0) debug(msg);
	  throw new IOException(msg);
	  }
*/
        msg = "********************* New value: XXX";
        if (DEBUG && debuglevel > 0) debug(msg);
        obj.setValue(value);

        if (DEBUG) trace(OUT, "...decodeExplicitTaggedType()" );
    }

    /** Decodes an IMPLICIT Tagged Type. */
    public boolean decodeImplicitTaggedType(ASNTaggedType obj, InputStream in)
    throws IOException {
        if (DEBUG) trace(IN, "...decodeImplicitTaggedType("+obj+")");

        byte[] origTag = Tag.getTag(in);

        Tag tag = obj.getTag();
        if (tag.isExplicit()) {
            if (DEBUG) trace(OUT, "...decodeImplicitTaggedType() --> false");
            return false;
        }

        int length = decodeLength(in);
        Tag childTag;

        // Ok. We've got to synthesize something that never existed.
        // The buffer that would have been here if the IMPLICIT
        // hadn't stomped our tag.
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        if (obj.jjtGetNumChildren() != 1) {
            String msg = "Implicitly Tagged types must have exactly one child";
            if (DEBUG && debuglevel > 8) debug(msg);
            throw new IOException(msg);
        }

        childTag = obj.getChild(0).getTag();
        if (childTag == null) {
            if (DEBUG && debuglevel > 0)
                debug("... decoding an IMPLICIT ANY, using original tag");
            os.write(origTag);
        } else
            os.write(childTag.getValue());

        encodeLength(length, os);
        byte[] TL = os.toByteArray();
        byte[] buffer = new byte[length+TL.length];
        System.arraycopy(TL, 0, buffer, 0, TL.length);
        int actualLength = in.read(buffer, TL.length, length);

        if (actualLength == -1)
            throw new java.io.EOFException();
        if (actualLength != length)
            throw new IOException("Length ("+length+") mismatch: "+actualLength);

        // now parse the contents of this constructed Tagged
        // element into its components
        CoderOperations ber = BaseCoder.getInstance("DER");
        ber.init(new ByteArrayInputStream(buffer));

        Object value = obj.childrenAccept(ber, null);
        obj.setValue(value);

        if (DEBUG) trace(OUT, "...decodeImplicitTaggedType() --> true");
        return true;
    }


// stream- methods
//...........................................................................

    public void decode(InputStream in, int level)
    throws IOException {
        if (DEBUG) trace(IN, "decode");

        Vector entries = new Vector();
        Tag tag;
        int length;
        byte[] buffer;
        while (in.available() != 0)
            try {
                tag = Tag.decode(in);
                length = decodeLength(in);
                buffer = new byte[length];
                in.read(buffer);
                entries.addElement(new Component(tag, length, buffer));
            } catch (java.io.EOFException x) {
                x.printStackTrace(err);
                break;
            }

        if (DEBUG && debuglevel > 8) { // dump the lot
            String prefix = "\t";
            for (int i = 0; i < level; i++)
                prefix += "+--";
            level++;
            for (int i = 0; i < entries.size(); i++) {
                Component x = (Component) entries.elementAt(i);
                tag = x.getTag();
                length = x.getLength();
                buffer = x.getData();
                System.out.println(prefix+"Tag: "+tag+ "; Length: "+length);
                if (tag.isConstructed())
                    decode(new java.io.ByteArrayInputStream(buffer), level);
            }
        }

        if (DEBUG) trace(OUT, "decode");
    }

    /** Encodes the length element of a DER data triplet. */
    private static void encodeLength(int length, OutputStream out)
    throws IOException {
        if (length < 128) { // short definite form
            out.write((byte) length);
            return;
        }
        if (length < 256) { // long definite form
            out.write(-127);
            out.write((byte) length);
            return;
        }
        if (length < 65536) {
            out.write(-126);
            out.write((byte)(length >> 8));
            out.write((byte) length);
            return;
        }
        if (length < 16777216) {
            out.write(-125);
            out.write((byte)(length >> 16));
            out.write((byte)(length >>  8));
            out.write((byte) length);
            return;
        }
        out.write(-124);
        out.write((byte)(length >> 24));
        out.write((byte)(length >> 16));
        out.write((byte)(length >>  8));
        out.write((byte) length);
    }

    private byte[] getTLV(InputStream in)
    throws IOException {
        ByteArrayOutputStream bos=new ByteArrayOutputStream();
        byte[] tagbytes = Tag.getTag(in);
        if (DEBUG && debuglevel > 0) debug("Tag: "+Hex.toString(tagbytes));
        int length = decodeLength(in);
        bos.write(tagbytes);
        encodeLength(length,bos);
        byte[] TL = bos.toByteArray();
        byte[] buffer = new byte[TL.length+length];
        System.arraycopy(TL, 0, buffer, 0, TL.length);
        if (length > 0) {
            int actualLength = in.read(buffer, TL.length, length);
            if (actualLength == -1)
                throw new EOFException();
            if (actualLength != length)
                throw new IOException("Length ("+length+") mismatch: "+actualLength);
        }

        return buffer;
     }


// Inner classes
//...........................................................................

    class Component
    {
        Tag tag;
        int length;
        byte[] data;

        Component(Tag tag, int length, byte[] data) {
            this.tag = tag;
            this.length = length;
            this.data = (byte[]) data.clone();
        }

        Tag getTag() {
            return tag;
        }

        int getLength() {
            return length;
        }

        byte[] getData() {
            return (byte[]) data.clone();
        }
    }

//-----------------------------------------------------------------
// Copyright
// Sun Microsystems, Inc.
//
//
// Copyright (C) 1994, 1995 Sun Microsystems, Inc.
// All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person
// obtaining a copy of this software and associated documentation
// files (the "Software"), to deal in the Software without
// restriction, including without limitation the rights to use,
// copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software or derivatives of the Software, and to 
// permit persons to whom the Software or its derivatives is furnished 
// to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
// OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT.// IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
// THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Sun Microsystems, Inc.
// shall not be used in advertising or otherwise to promote
// the sale, use or other dealings in this Software or its derivatives 
// without prior written authorization from Sun Microsystems, Inc.

}
