/* $Id: $
 *
 * Copyright (c) 1997, 1998, 1999 Systemics Ltd on behalf of
 * the Cryptix Development Team.  All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General License along with this library; if
 * not, you can download a copy from <http://www.cryptix.org/>.
 */

/**
 * Originally generated by <code>jjTree</code> (part of the
 * <code>JavaCC</code> tool), this Class acts as the concrete
 * implementation of a <code>cryptix.asn1.lang.Node</code> or
 * an <code>cryptix.asn1.lang.ASNObject</code> in a jjtree output
 * structure.<p>
 *
 * <b>Copyright</b> &copy;1997, 1998, 1999
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: 1.1 $</b>
 * @author  Raif S. Naffah
 */

package cryptix.asn1.lang;

import java.io.PrintWriter;
import java.util.Vector;

public class SimpleNode
implements ASNObject
{
// Debugging methods and fields
//...........................................................................

    private static final String NAME = "SimpleNode";

    private static final boolean IN = true, OUT = false;
    private static final boolean DEBUG =
        PackageProperties.GLOBAL_DEBUG;
    private static final int debuglevel =
        DEBUG ? PackageProperties.getLevel(NAME) : 0;
    private static final PrintWriter err =
        DEBUG ? PackageProperties.getOutput() : null;
    private static final boolean TRACE =
        PackageProperties.isTraceable(NAME);

    private static void debug(String s) {
        err.println(">>> "+NAME+": "+s);
    }

    private static void trace(boolean in, String s) {
        if (TRACE) err.println((in?"==> ":"<== ")+NAME+"."+s);
    }

    private static void trace(String s) {
        if (TRACE) err.println("<=> "+NAME+"."+s);
    }


// Constants and variables
//...........................................................................

    protected ASNObject parent;
    protected ASNObject[] children;
    protected int id;
    protected Parser parser;


// Constructors and factory methods
//...........................................................................

    public SimpleNode(int i) {
        id = i;
    }

    public SimpleNode(Parser p, int i) {
        this(i);
        parser = p;
    }

    /**
     * Returns a new instance given a UNIVERSAL ASN.1 tag instance;
     * null otherwise.
     */
    public static final SimpleNode getInstance(Parser p, Tag tag) {
        int clazz = tag.getClazz();
        if (clazz != Tag.UNIVERSAL)
            return null;
        int value = tag.getValue();
        int i;
        SimpleNode result = null;
        switch (value) {
        case Tag.BOOLEAN:
            result = new ASNBoolean(p, ParserTreeConstants.JJTBOOLEAN); break;
        case Tag.INTEGER:
            result = new ASNInteger(p, ParserTreeConstants.JJTINTEGER); break;
        case Tag.BIT_STRING:
            result = new ASNBitString(p, ParserTreeConstants.JJTBITSTRING); break;
        case Tag.OCTET_STRING:
            result = new ASNOctetString(p, ParserTreeConstants.JJTOCTETSTRING); break;
        case Tag.NULL:
            result = new ASNNull(p, ParserTreeConstants.JJTNULL); break;
        case Tag.OBJECT_IDENTIFIER:
            result = new ASNObjectIdentifier(p, ParserTreeConstants.JJTOBJECTIDENTIFIER); break;
        case Tag.SEQUENCE:
            result = new ASNSequence(p, ParserTreeConstants.JJTSEQUENCE); break;
        case Tag.SEQUENCE_OF:
            result = new ASNSequenceOf(p, ParserTreeConstants.JJTSEQUENCEOF); break;
        case Tag.SET:
            result = new ASNSet(p, ParserTreeConstants.JJTSET); break;
        case Tag.SET_OF:
            result = new ASNSetOf(p, ParserTreeConstants.JJTSETOF); break;
        case Tag.PRINT_STRING:
        case Tag.T61_STRING:
        case Tag.IA5_STRING:
            result = new ASNPrintableString(p, ParserTreeConstants.JJTPRINTABLESTRING); break;
        case Tag.UTC_TIME:
            result = new ASNTime(p, ParserTreeConstants.JJTTIME); break;
        }
        return result;
    }


// jjTree methods and hooks
//...........................................................................

    public void jjtOpen() {
    }

    public void jjtClose() {
    }

    public void jjtSetParent(Node n) {
        parent = (ASNObject) n;
    }

    public Node jjtGetParent() {
        return parent;
    }

    public ASNObject getParent() {
        SimpleNode result = this;
        do {
            result = (SimpleNode) result.jjtGetParent();
        } while (result != null && result.id == ParserTreeConstants.JJTTYPE);
        return (ASNObject) result;
    }

    public void jjtAddChild(Node n, int i) {
        if (children == null)
            children = new ASNObject[i + 1];
        else if (i >= children.length) {
            ASNObject[] c = new ASNObject[i + 1];
            System.arraycopy(children, 0, c, 0, children.length);
            children = c;
        }
        children[i] = (ASNObject) n;
    }

    public Node jjtGetChild(int i) {
        return children[i];
    }

    public ASNObject getChild(int i) {
        return (ASNObject) jjtGetChild(i);
    }

    public ASNObject[] getChildren() {
        return children;
    }

    public int jjtGetNumChildren() {
        return children == null ? 0 : children.length;
    }

    /** Accepts the visitor. */
    public Object jjtAccept(ParserVisitor visitor, Object data)
    throws java.io.IOException {
        return visitor.visit(this, data);
    }

    /** Accepts the visitor. */
    public Object childrenAccept(ParserVisitor visitor, Object data)
    throws java.io.IOException {
        Vector result = new Vector();
        if (children != null)
            for (int i = 0; i < children.length; ++i) {
                children[i].jjtAccept(visitor, data);
                result.addElement(children[i].getValue());
            }
        return result;
    }

    /**
     * You can override these two methods in subclasses of SimpleNode
     * to customize the way the node appears when the tree is dumped.
     * If your output uses more than one line you should override
     * toString(String), otherwise overriding toString() is probably
     * all you need to do.
     */
    public String toString() {
        String result = "<ASNObject name=\""+name+"\" type=\"";
        result += ParserTreeConstants.jjtNodeName[id]+"\"";
        result += " default=\""+defaultValue+"\"";
//        if (value != null)
//            result += " value=\""+value+"\"";
        result += " optional=\""+(optional ? "yes" : "no")+"\" />";
        return result;
    }

    public String toString(String prefix) {
        String result = prefix+toString();
        return result;
    }


// Own variables and methods: specific ASN.1 handling
//...........................................................................

    /** Name associated with this ASN.1 object. */
    protected String name;

    /** Tag associated with this ASN.1 object. */
    protected Tag tag;

    /** True if the ASN.1 object is optional, false otherwise. */
    protected boolean optional;

    /**
     * Java object representing the current value of this ASN.1 object
     * when applicable. For the Java types used to represent these
     * values see <code>cryptix.asn1.lang.ASNObject</code>.
     *
     * @see cryptix.asn1.lang.ASNObject#setValue
     */
    protected Object value;

    /**
     * The Java object representing the default value for this
     * ASN.1 object if such value is defined in the specification.
     */
    protected Object defaultValue;

    /**
     * @return The <code>cryptix.asn1.lang.Parser</code> handling
     *            this ASN.1 object.
     */
    public Parser getParser() {
        return parser;
    }

    /** @return The ID assigned to this ASN.1 Object by the Parser. */
    public int getID() {
        return id;
    }


// ASNObject interface implementation
//...........................................................................

    /** Sets the name of this instance. */
    public void setName(String name) {
        this.name = name;
    }

    /** @return The name associated with this ASN.1 object. */
    public String getName() {
        return name;
    }

    public ASNObject getComponent(String aName) {
        if (DEBUG) trace(IN, "getComponent");
        if (DEBUG && debuglevel > 8) debug(" IN: \""+aName+"\"");

        ASNObject result = (ASNObject) parser.resolve(aName);

        if (DEBUG && debuglevel > 8) debug("OUT: "+result);
        if (DEBUG) trace(OUT, "getComponent");
        return result;
    }

    public void setTag(Tag tag) {
        this.tag = tag;
    }

    public Tag getTag() {
        return tag;
    }

    public boolean isOptional() {
        return optional;
    }

    public void setOptional(boolean flag) {
        optional = flag;
    }

    public void setValue(Object value) {
        if (this.value != null)
            if (DEBUG && debuglevel > 1) debug("Is already assigned: "+value);
        this.value = value;
    }

    public Object getValue() {
        Object result = null;
        if (id == ParserTreeConstants.JJTTYPEALIAS) {
            SimpleNode n = (SimpleNode) parser.resolve(name);
            if (n != null)
                result = n.getValue();
        } else
            result = value;

        return result;
    }

    public void setDefaultValue(Object defaultValue) {
        this.defaultValue = defaultValue;
    }

    public Object getDefaultValue() {
//        return defaultValue;
        if (this instanceof ASNType) {
            SimpleNode x = (SimpleNode) children[0];
            return x.defaultValue;
        }
        else if (this instanceof ASNTaggedType) {
            SimpleNode x = (SimpleNode) parser.resolve(name);
            return x.defaultValue;
        }
        else if (this instanceof ASNTypeAlias) {
            SimpleNode x = (SimpleNode) parser.resolve(name);
            return x.defaultValue;
        }
        else
            return defaultValue;
    }

    public void dump() {
        System.out.println(toString());
    }

    public void dump(String prefix) {
        if (this instanceof ASNType)
            dumpType(this, prefix);
        else if (this instanceof ASNTaggedType)
            dumpTaggedType(this, prefix);
        else if (this instanceof ASNTypeAlias)
            dumpTypeAlias(this, prefix);
        else
            dumpSimpleNode(this, prefix);
    }

    public Object accept(ParserVisitor visitor, Object data)
    throws java.io.IOException {
        return this.jjtAccept(visitor, data);
    }


// helper methods
//...........................................................................

    private void dumpType(SimpleNode n, String prefix) {
        if (DEBUG && debuglevel > 8) prefix += "T.";
        SimpleNode x = (SimpleNode) n.children[0];
        x.dump(prefix);
    }

    private void dumpTaggedType(SimpleNode n, String prefix) {
        if (DEBUG && debuglevel > 8) prefix += "TT.";
        SimpleNode x = (SimpleNode) parser.resolve(n.name);
        System.out.println(x.toString(prefix+n.getTag()));
    }

    private void dumpTypeAlias(SimpleNode n, String prefix) {
        if (DEBUG && debuglevel > 8) prefix += "A.";
        SimpleNode x = (SimpleNode) parser.resolve(n.name);
        x.dump(prefix);
    }

    private void dumpSimpleNode(SimpleNode n, String prefix) {
        if (DEBUG && debuglevel > 8) prefix += "S.";
        System.out.println(n.toString(prefix));
        dumpChildren(this.children, prefix);
    }

    private void dumpChildren(ASNObject[] children, String prefix) {
        if (children == null)
            return;
        prefix += "+ ";
        int limit = children.length;
        for (int i = 0; i < limit; ++i) {
            SimpleNode x = (SimpleNode) children[i];
            if (x != null)
                x.dump(prefix);
        }
    }
}
