/* $Id: $
 *
 * Copyright (c) 1997, 1998, 1999 Systemics Ltd on behalf of
 * the Cryptix Development Team.  All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have 
 * received a copy of the Cryptix General License along with this library; 
 * if not, you can download a copy from <http://www.cryptix.org/>.
 */

package cryptix.asn1.lang;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.PushbackInputStream;

/**
 * A class to handle ASN.1 Tag elements.<p>
 *
 * <b>Copyright</b> &copy;1997, 1998, 1999
 * <a href="http://www.systemics.com/">Systemics Ltd</a> on behalf of the
 * <a href="http://www.systemics.com/docs/cryptix/">Cryptix Development Team</a>.
 * <br>All rights reserved.<p>
 *
 * <b>$Revision: 1.3 $</b>
 * @author  Raif S. Naffah
 * @author  Eric Rescorla
 */
public class Tag
{
// Debugging methods and fields
//...........................................................................

    private static final String NAME = "Tag";

    private static final boolean IN = true, OUT = false;
    private static final boolean DEBUG =
        PackageProperties.GLOBAL_DEBUG;
    private static final int debuglevel =
        DEBUG ? PackageProperties.getLevel(NAME) : 0;
    private static final PrintWriter err =
        DEBUG ? PackageProperties.getOutput() : null;
    private static final boolean TRACE =
        PackageProperties.isTraceable(NAME);

    private static void debug(String s) {
        err.println(">>> "+NAME+": "+s);
    }

    private static void trace(boolean in, String s) {
        if (TRACE) err.println((in?"==> ":"<== ")+NAME+"."+s);
    }

    private static void trace(String s) {
        if (TRACE) err.println("<=> "+NAME+"."+s);
    }


// Constants and variables
//...........................................................................

    public static final int UNIVERSAL   = 0x00;
    public static final int APPLICATION = 0x40;
    public static final int CONTEXT     = 0x80;
    public static final int PRIVATE     = 0xC0;

    public static final int BOOLEAN           = 0x01;
    public static final int INTEGER           = 0x02;
    public static final int BIT_STRING        = 0x03;
    public static final int OCTET_STRING      = 0x04;
    public static final int NULL              = 0x05;
    public static final int OBJECT_IDENTIFIER = 0x06;
    public static final int SEQUENCE          = 0x10;
    public static final int SEQUENCE_OF       = 0x30;
    public static final int SET               = 0x11;
    public static final int SET_OF            = 0x31;
    public static final int PRINT_STRING      = 0x13;
    public static final int T61_STRING        = 0x14;
    public static final int IA5_STRING        = 0x16;
    public static final int UTC_TIME          = 0x17;

    int clazz; // the tag's class id
    int value; // the tag's actual value
    boolean explicit; // is it explicit or implicit?
    boolean constructed; // is it constructed?


// Constructors
//...........................................................................

    /**
     * Constructs an ASN.1 Tag instance.
     *
     * @param clazz  The tag's class, default is UNIVERSAL.
     * @param value  The tag's value.
     * @param explicit  Whether this tag is explicit or implicit.
     * @param constructed  Whether this tag is constructed or not.
     *        Default is not constrcuted.
     */
    Tag(int clazz, int value, boolean explicit, boolean constructed) {
        this.clazz = clazz;
        this.value = value;
        this.explicit = explicit;
        this.constructed = constructed;
    }

    /**
     * Convenience constructor. Constructs an ASN.1 Tag instance.
     *
     * @param clazz  The tag's class, default is UNIVERSAL.
     * @param value  The tag's value.
     * @param explicit  Whether this tag is explicit or implicit.
     */
    Tag(int clazz, int value, boolean explicit) {
        this(clazz, value, explicit, false);
    }

    /**
     * Convenience constructor. Constructs an ASN.1 Tag instance.
     *
     * @param value  The tag's value.
     * @param explicit  Whether this tag is explicit or implicit.
     */
    Tag(int value, boolean explicit) {
        this(Tag.UNIVERSAL, value, explicit);
    }


// Accessor methods
//...........................................................................

    /** Returns the tag's class. */
    public int getClazz() {
        return clazz;
    }

    /** Returns the tag's class number. */
    public int getValue() {
        return value;
    }

    /** Returns true if the tag is explicit, false otherwise. */
    public boolean isExplicit() {
        return explicit;
    }

    /** Returns true if the tag is constructed false otherwise. */
    public boolean isConstructed() {
        return constructed;
    }


// Utility methods and factories
//...........................................................................

    /**
     * Returns the tag element of an ASN.1 object if it is of the
     * designated expected class and type, or null otherwise. If the
     * stream does not contain the expected tag (of given type) this
     * method ensures that the stream marker is not modified.
     *
     * @param expectedClass The tag's class expected to be found at
     *        the current marker location of the given input stream.
     * @param expectedValue The tag's number expected to be found at
     *        the current marker location of the given input stream.
     * @param in The input source stream.
     * @return An instance of the <code>Tag</code> class containing
     *        the concrete <code>Tag</code> instance found in the
     *        input stream.
     */
    public static Tag
    getExpectedTag(int expectedClass, int expectedValue, InputStream in)
    throws IOException {
        Tag result = null;
        ByteArrayOutputStream recovery = new ByteArrayOutputStream();
        int tagClass = -1;
        int tagValue = -1;
        try {
            int c = in.read();
            if (c < 0)
                return null;
            c &= 0xFF;
            recovery.write(c);
            tagClass = c & 0xC0	;
            boolean tagConstructed = (c & 0x20) != 0;
            tagValue = c & 0x1F;
            if (tagValue == 0x1F) { // multiple bytes for tag number
                tagValue = 0;
                c = in.read();
                if (c<0)
                    return null;
                c &= 0xFF;
                do { // sum all following bytes with set MSB
                    tagValue += c & 0x80;
                    c = in.read();
                    if (c < 0)
                        return null;
                    c &= 0xFF;
                    recovery.write(c);
                } while ((c & 0x80) != 0);
            }
            result = new Tag(tagClass, tagValue, true, tagConstructed);
            if (DEBUG && debuglevel > 6)
                debug("Checking for Tag's class=\""+expectedClass+"\", value \""+
                    expectedValue+"\"; found: "+result);
        } finally {
            if (!eval(tagClass, expectedClass, tagValue, expectedValue)) {
                result = null;
                if (DEBUG && debuglevel > 6) debug("Recovering...");
                byte[] recovered = recovery.toByteArray();
                PushbackInputStream pbis =
                    in instanceof PushbackInputStream ?
                        (PushbackInputStream) in :
                        new PushbackInputStream(in, 2048);
                pbis.unread(recovered);
                in = pbis;
            }
            recovery = null;
        }

        return result;
    }

    /**
     * Convenience method similar to the method with same name and 3
     * arguments, except it assumes that the tag's class is UNIVERSAL.
     *
     * @param expectedValue The tag's number expected to be found at
     *        the current marker location of the given input stream.
     * @param in The input source stream.
     * @return An instance of the <code>Tag</code> class containing
     *        the concrete <code>Tag</code> instance found in the
     *        input stream.
     */
    public static Tag getExpectedTag(int expectedValue, InputStream in)
    throws IOException {
        return getExpectedTag(UNIVERSAL, expectedValue, in);
    }

    /**
     * Convenience method similar to the method with same name and 3
     * arguments, except it uses the given tag's class and value.
     *
     * @param tag The tag's instance (class and value) expected to be
     *        found at the current marker location of the given input
     *        stream.
     * @param in The input source stream.
     * @return An instance of the <code>Tag</code> class containing
     *        the concrete <code>Tag</code> instance found in the
     *        input stream.
     */
    public static Tag getExpectedTag(Tag tag, InputStream in)
    throws IOException {
        return getExpectedTag(tag.getClazz(), tag.getValue(), in);
    }

    /**
     * Get a tag off the wire as a <code>byte[]</code>
     * @param in The input stream
     * @return a <code>byte</code> with the tag bytes
     * @exception IOException
     * --EKR
     */
    public static byte[] getTag(InputStream in)
    throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        int c = in.read() & 0xFF;
        bos.write(c);
        c &= 0xFF;
        if ((c & 0x1f) == 0x1F)
            do {
                c = in.read() & 0xFF;
                bos.write(c);
            } while ((c & 0x80) != 0);

        return bos.toByteArray();
    }
	 
    /**
     * Constructs a <code>Tag</code> instance from the designated
     * input stream.
     *
     * @param in The input stream.
     * @return A <code>Tag</code> instance parsed from an input
     *        stream.
     * @exception IOException If an error occurs while parsing
     *        the input stream.
     */
    public static Tag decode(InputStream in)
    throws IOException {
        int c = in.read() & 0xFF;
        int tagClass = c & 0xC0;
        boolean tagConstructed = (c & 0x20) != 0;
        int tagNumber = c & 0x1F;
        if (tagNumber == 0x1F) { // multiple bytes for tag number
            tagNumber = 0;
            c = in.read() & 0xFF;
            do { // sum all following bytes with set MSB
                tagNumber += c & 0x80;
                c = in.read() & 0xFF;
            } while ((c & 0x80) != 0);
        }

        Tag result = new Tag(tagClass, tagNumber, true, tagConstructed);
        if (DEBUG && debuglevel > 6) debug("Decoded "+result);
        return result;
    }

    /**
     * Returns the Tag instance parsed from the given input stream,
     * if one is there, without modifying the stram marker.
     *
     * @param in The input stream.
     * @return A <code>Tag</code> instance parsed from the designated
     *        input stream.
     * @exception IOException If an error occurs while parsing
     *        the input stream.
     */
    public static Tag peek(InputStream in)
    throws IOException {
        in.mark(Integer.MAX_VALUE);
        Tag result = null;
        try {
            result = decode(in);
        } finally {
            in.reset();
        }

        return result;
    }


// helper methods
//...........................................................................

    private static boolean eval(int tagClass, int expectedClass,
                                int tagValue, int expectedValue) {
        if (tagClass != expectedClass)
            return false;
        if (tagClass != UNIVERSAL)
            return tagValue == expectedValue;
        if (expectedValue == PRINT_STRING || expectedValue == IA5_STRING)
            return tagValue == PRINT_STRING || tagValue == IA5_STRING;
        return tagValue == expectedValue;
    }


// Visualisation methods
//...........................................................................

    public String toString() {
        String result = "<Tag class=\"";
        switch (clazz) {
        case UNIVERSAL:   result += "UNIVERSAL ("+UNIVERSAL+")"; break;
        case APPLICATION: result += "APPLICATION ("+APPLICATION+")"; break;
        case CONTEXT:     result += "CONTEXT ("+CONTEXT+")"; break;
        case PRIVATE:     result += "PRIVATE ("+PRIVATE+")"; break;
        default:          result += ""+clazz;
        }

        result += "\" value=\"";
        if (clazz == CONTEXT)
            result += ""+value;
        else
            switch (value) {
            case BOOLEAN:           result += "BOOLEAN ("+BOOLEAN+")"; break;
            case INTEGER:           result += "INTEGER ("+INTEGER+")"; break;
            case BIT_STRING:        result += "BIT STRING ("+BIT_STRING+")"; break;
            case OCTET_STRING:      result += "OCTET STRING ("+OCTET_STRING+")"; break;
            case NULL:              result += "NULL ("+NULL+")"; break;
            case OBJECT_IDENTIFIER: result += "OBJECT_IDENTIFIER ("+OBJECT_IDENTIFIER+")"; break;
            case SEQUENCE:          result += "SEQUENCE ("+SEQUENCE+")"; break;
            case SET:               result += "SET ("+SET+")"; break;
            case PRINT_STRING:      result += "PrintableString ("+PRINT_STRING+")"; break;
            case T61_STRING:        result += "T61String ("+T61_STRING+")"; break;
            case IA5_STRING:        result += "IA5String ("+IA5_STRING+")"; break;
            case UTC_TIME:          result += "UTCTime ("+UTC_TIME+")"; break;
            default:                result += ""+value;
            }

        result += "\" explicit=\""+(explicit ? "yes" : "no");
        result += "\" constructed=\""+(constructed ? "yes" : "no")+"\" />";
        return result;
    }
}

