/*
 * $COPYRIGHT$
 * $Id: SepInterpreter.java,v 1.47 2002/05/31 21:05:58 crow Exp $
 *
 * Date         Author          Changes
 * JUL 11 2001  Szabo Csaba     Created
 */

package org.media.datastore.sepengine;

import antlr.*;

import java.io.*;
import java.util.*;
import java.net.URL;
import org.xml.sax.InputSource;

import org.media.datastore.sepengine.*;
import org.media.datastore.sepengine.parser.*;
import org.media.datastore.sepengine.util.*;
import org.media.datastore.sepengine.LogManager;

import org.beepcore.beep.profile.sasl.otp.*; 
import org.beepcore.beep.profile.sasl.otp.algorithm.md5.*;
import org.beepcore.beep.profile.sasl.otp.algorithm.sha1.*;

/**
 * @author <a href="mailto:crow@nolimits.ro">Szabo Csaba </a>
 * @version $Revision: 1.47 $ $Date: 2001/08/21 15:00:00
 * @see java.lang.String
 * @see java.io.InputStream
 * @see org.media.datastore.sepengine.RDBInfo
 */
     
public class SepInterpreter {
    
    private static final int        BLOCK_SIZE = 4000;
    private static final String     hashMethod = "otp-md5";
    private static final String           seed = "xyz";
    private static final int               seq = 1;

    public  Vector          ignoredKeys = new Vector();   
    private SepDBHandler        handler = null;
    private SepADMDBHandler  admHandler = null;
    private String             errorMsg = "";
    private StringBuffer         header = new StringBuffer();
    private RDBInfo             rdbInfo = null;
    private int                 blockNr = 0;
    private String                reqNr = "0";    
    

    public SepInterpreter() {
    }
  
  
    public String preDigest( String pwd ) throws Exception {
        return SASLOTPProfile.convertBytesToHex(
						( new MD5() ).generateHash( pwd.getBytes() )
						);
    }

    
    public String getPasswdHash( String pwd ) throws Exception {
        byte[] pw = ( seed + pwd ).getBytes();
                                                     
        try {
            for ( int cx = 0; cx <= seq; cx++ ) {
                if ( hashMethod.equals("otp-md5") )
                    pw = ( new MD5() ).generateHash( pw );
                else
                    if ( hashMethod.equals("otp-sha1") )
                        pw = ( new SHA1() ).generateHash( pw );
                    else
                        throw new Exception("\nInvalid algorithm method:" + hashMethod + "\n");
            }
        }
        catch ( Exception e ) {
            throw e;
        }

        return SASLOTPProfile.convertBytesToHex( pw );
    }
    
    
    public String getHashMethod() {
        return hashMethod;
    }

    
    public String getSeed() {
        return seed;
    }
    

    public String getSeq() {
        return seq + "";
    }

    
    public String getErrorMSG() {
        return errorMsg;
    }
    
    
    public String getRequestNr() {
        return reqNr;
    }
    
    
    public InputStream execute (String str, RDBInfo info, String uname) {
        return execute( new ByteArrayInputStream( str.getBytes() ), info , uname);
    }

    
    public String getResponse( String type, int codeNr, String msg ) {
        if ( type.equalsIgnoreCase("info") )
            return "<response reqno='" + reqNr + "'>\n" +
                "<additional>\n" +
                "<info code=\"" + codeNr +"\">\n" + 
                msg + "\n" +
                "</info>\n" +
                "</additional>\n" +
                "</response>\n";
        else if ( type.equalsIgnoreCase("error") )
            return "<response reqno='" + reqNr + "'>\n" + 
                "<error code=\"" + codeNr +"\">\n" +
                msg + "\n" + 
                "</error>\n" +
                "</response>\n";
        else if ( type.equalsIgnoreCase("warn") )
            return "<response reqno='" + reqNr + "'>\n" +
                "<additional>\n" +
                "<warning code=\"" + codeNr +"\">\n" + 
                msg + "\n" +
                "</warning>\n" +
                "</additional>\n" +
                "</response>\n";

        return msg; 
    }


    public InputStream execute (InputStream istream, RDBInfo info, String uname) {
        StringInputStream sis = new StringInputStream(); 
        
        if ( info == null ) {
            sis.write( getResponse("error", 0, LogManager.getReference().getMessage("rdb_missing") ) );
            return sis.getInputStream();
        }
        String         result = LogManager.getReference().getMessage("not_sep");
        DTDDirInfo    dtdInfo = info.getDTDDirInfo(); 
        String           mode = "flexible";
        String         naming = "auto";
        String        docName = null;
        boolean      isUpdate = false;
        
        rdbInfo    = info;
        handler    = new SepDBHandler(    info, uname, this);
        admHandler = new SepADMDBHandler( info, uname, this);
        
        if ( dtdInfo.getStoreMode() != null )
            mode = dtdInfo.getStoreMode().trim();
        
        if ( dtdInfo.getNamingFlag() != null )
            naming = dtdInfo.getNamingFlag();

        if ( handler.isReady() ) {
            SepReturnTable ret = retTable(istream, uname);
            reqNr = ret.reqno + "";
            String sql = ret.sql;

            if ( sql == null ) sql = ret.error;

            if ( "fetch".equals( ret.type ) || ( ret.type == null ) )
                return executeFetch(sql, ret.scanTree);

            if ( "store".equals( ret.type ) ) {
                String    action = ret.action;
                String blockName = getValue( _nextToken(istream).trim() ).trim();

                if ( blockName.startsWith ("_admin.") ) {
                    InputStream _is = new ByteArrayInputStream( validBlock( istream ).getBytes() );
                
                    if ( blockName.startsWith("_admin.users") ) {
                        UserAnalyzer ua = new UserAnalyzer(); 
                        UserInfo  uInfo = ua.getUserInfo ( _is , new UserInfo() );
                        
                        if ( uInfo == null ) {
                            sis.write( getResponse("error", 0, ua.getErrorMSG() ) );
                            return sis.getInputStream();
                        }
                        
                        if ( uInfo.getAction().equalsIgnoreCase("add") ) {
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.addUser( uInfo ) ) {
                                    sis.write( getResponse("info", 0,  
                                                           LogManager.getReference().getMessage("user_added") ) );
                                    LogManager.getReference().message( "info", "add_user.ok", 
                                                                       " '" + uname + "'.", "'" + uInfo.getUserName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) ); 
                                    LogManager.getReference().message( "warn", "add_user.fail",
                                                                       " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0,
                                                       LogManager.getReference().getMessage("user_add_fail") ) );
                                LogManager.getReference().message( "warn", "add_user.fail",
                                                                   " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                            }
                        }
                        
                        if ( uInfo.getAction().equalsIgnoreCase("mod") ) { 
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.modUser( uInfo ) ) {
                                    sis.write( getResponse("info", 0, 
                                                           LogManager.getReference().getMessage("user_updated") ) );
                                    LogManager.getReference().message( "info", "mod_user.ok", 
                                                                       " '" + uname + "'.", "'" + uInfo.getLoginName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) );
                                    LogManager.getReference().message( "warn", "mod_user.fail",
                                                                       " '" + uInfo.getLoginName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0,
                                                       LogManager.getReference().getMessage("user_update_fail") ) );
                                LogManager.getReference().message( "warn", "mod_user.fail",
                                                                   " '" + uInfo.getLoginName() + "'.", " '" + uname + "' " ); 
                            }
                        }

                        if ( uInfo.getAction().equalsIgnoreCase("del") ) { 
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.delUser( uInfo ) ) {
                                    sis.write( getResponse("info", 0,  
                                                           LogManager.getReference().getMessage("user_del") ) );
                                    LogManager.getReference().message( "info", "del_user.ok", 
                                                                       " '" + uname + "'.", "'" + uInfo.getUserName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) );
                                    LogManager.getReference().message( "warn", "del_user.fail",
                                                                       " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0, 
                                                       LogManager.getReference().getMessage("user_del_fail") ) );
                                LogManager.getReference().message( "warn", "del_user.fail",
                                                                   " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                            }
                        }

                        if ( uInfo.getAction().equalsIgnoreCase("block") ) {
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.blockUser( uInfo ) ) { 
                                    sis.write( getResponse("info", 0, 
                                                           LogManager.getReference().getMessage("user_block") ) );
                                    LogManager.getReference().message( "info", "block_user.ok", 
                                                                       " '" + uname + "'.", "'" + uInfo.getUserName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, 
                                                           LogManager.getReference().getMessage("user_block_error") ) );
                                    LogManager.getReference().message( "warn", "block_user.fail",
                                                                       " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0,
                                                       LogManager.getReference().getMessage("user_block_fail") ) );
                                LogManager.getReference().message( "warn", "block_user.fail",
                                                                   " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                            }
                        }
                        
                        if ( uInfo.getAction().equalsIgnoreCase("ublock") ) {
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.unblockUser( uInfo ) ) {
                                    sis.write( getResponse("info", 0,  
                                                           LogManager.getReference().getMessage("user_ublock") ) );
                                    LogManager.getReference().message( "info", "ublock_user.ok", 
                                                                       " '" + uname + "'.", "'" + uInfo.getUserName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, 
                                                           LogManager.getReference().getMessage("user_ublock_error") ) );
                                    LogManager.getReference().message( "warn", "ublock_user.fail",
                                                                       " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0,
                                                       LogManager.getReference().getMessage("user_ublock_fail") ) );
                                LogManager.getReference().message( "warn", "ublock_user.fail",
                                                                   " '" + uInfo.getUserName() + "'.", " '" + uname + "' " ); 
                            }
                        }
                    }
                    
                    if ( blockName.startsWith("_admin.groups") ) {
                        GroupAnalyzer  ga = new GroupAnalyzer();
                        GroupInfo   gInfo = ga.getGroupInfo ( _is , new GroupInfo() );
                        
                        if ( gInfo == null ) {
                            sis.write( getResponse("error", 0, ga.getErrorMSG() ) );
                            return sis.getInputStream();
                        }

                        if ( gInfo.getAction().equalsIgnoreCase("add") ) {
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.addGroup( gInfo ) ) {
                                    sis.write( getResponse("info", 0, 
                                                           LogManager.getReference().getMessage("group_added") ) );
                                    LogManager.getReference().message( "info", "add_group.ok", 
                                                                       " '" + uname + "'.", "'" + gInfo.getGroupName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) );
                                    LogManager.getReference().message( "warn", "add_group.fail",
                                                                       " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0, 
                                                       LogManager.getReference().getMessage("group_add_fail") ) );
                                LogManager.getReference().message( "warn", "add_group.fail",
                                                                   " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                            }
                        }
                        
                        if ( gInfo.getAction().equalsIgnoreCase("mod") ) { 
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.modGroup( gInfo ) ) {
                                    sis.write( getResponse("info", 0, 
                                                           LogManager.getReference().getMessage("group_updated") ) );
                                    LogManager.getReference().message( "info", "mod_group.ok", 
                                                                       " '" + uname + "'.", "'" + gInfo.getGroupName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) );
                                    LogManager.getReference().message( "warn", "mod_group.fail",
                                                                       " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0, 
                                                       LogManager.getReference().getMessage("group_update_fail") ) );
                                LogManager.getReference().message( "warn", "mod_group.fail",
                                                                   " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                            }
                        }

                        if ( gInfo.getAction().equalsIgnoreCase("del") ) { 
                            if ( admHandler.hasRights() ) {
                                if ( admHandler.delGroup( gInfo ) ) { 
                                    sis.write( getResponse("info", 0, 
                                                           LogManager.getReference().getMessage("group_del") ) );
                                    LogManager.getReference().message( "info", "del_group.ok", 
                                                                       " '" + uname + "'.", "'" + gInfo.getGroupName() + "' " );
                                }
                                else {
                                    sis.write( getResponse("error", 0, admHandler.getErrorMSG() ) );
                                    LogManager.getReference().message( "warn", "del_group.fail",
                                                                       " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                                }
                            }
                            else {
                                sis.write( getResponse("error", 0, 
                                                       LogManager.getReference().getMessage("group_del_fail") ) );
                                LogManager.getReference().message( "warn", "del_group.fail",
                                                                   " '" + gInfo.getGroupName() + "'.", " '" + uname + "' " ); 
                            }
                        }
                    }
                    
                    if ( blockName.startsWith("_admin.trees.owner") )
                        return admHandler.chown( ( new ChownAnalyzer() ).getChownInfo( _is, new ChownInfo() ) );

                    if ( blockName.startsWith("_admin.trees.mode") )
                        return admHandler.chmod( ( new ChmodAnalyzer() ).getChmodInfo( _is, new ChmodInfo() ) );
                    
                    if ( blockName.startsWith("_admin.trees") ) {
                        TreeAnalyzer ta = new TreeAnalyzer();
                        TreeInfo  tInfo = ta.getTreeInfo ( _is, new TreeInfo() );
                        
                        if ( tInfo == null ) {
                            sis.write( getResponse("error", 0, ta.getErrorMSG() ) );
                            return sis.getInputStream();
                        }
        
                        if ( tInfo.getAction().equalsIgnoreCase("create") ) {
                            if ( handler.createNewTree( tInfo.getSubtreeName(), true ) ) {
                                LogManager.getReference().message( "info", "create_subtree.ok", 
                                                                   " '" + uname + "'.", "'" + tInfo.getSubtreeName() + "' " );
                                sis.write ( getResponse("info", 0, 
                                                        tInfo.getSubtreeName() + " " + 
                                                        LogManager.getReference().getMessage("subtree_create") ) );
                            }
                            else {
                                LogManager.getReference().message( "warn", "create_subtree.fail", 
                                                                   " '" + tInfo.getSubtreeName() + "'.", "'" + uname + "' " );
                                sis.write ( getResponse("error", 0, 
                                                        LogManager.getReference().getMessage("subtree_create_failed") + " " +
                                                        tInfo.getSubtreeName() + ". " + handler.getErrorMessage() ) );
                            }
                        }
                        
                        if ( tInfo.getAction().equalsIgnoreCase("delete") ) {
                            if ( admHandler.removeTree( tInfo.getSubtreeName(), tInfo.isForced() ) ) {
                                LogManager.getReference().message( "info", "del_subtree.ok", 
                                                                   " '" + uname + "'.", "'" + tInfo.getSubtreeName() + "' " );
                                sis.write ( getResponse("info", 0, 
                                                        tInfo.getSubtreeName() + " " + 
                                                        LogManager.getReference().getMessage("subtree_remove") ) );
                            }
                            else {
                                LogManager.getReference().message( "warn", "del_subtree.fail", 
                                                                   " '" + tInfo.getSubtreeName() + "'.", "'" + uname + "' " );
                                sis.write ( getResponse("error", 0, 
                                                        LogManager.getReference().getMessage("subtree_remove_failed") + " " +
                                                        tInfo.getSubtreeName() + ". " + admHandler.getErrorMSG() ) );
                            }
                        }
                        
                        if ( tInfo.getAction().equalsIgnoreCase("list") ) {
                            return admHandler.listTree( tInfo.getSubtreeName(), tInfo.isLongFormat(), tInfo.isTreeList() );
                        }
                        
                        if ( tInfo.getAction().equalsIgnoreCase("copy") ) {
                            return admHandler.copyTree( tInfo.getSource(), tInfo.getDestination(), 
                                                        tInfo.isVerboseMode(), tInfo.isRecursive() );
                        }
                        
                        if ( tInfo.getAction().equalsIgnoreCase("chtype") ) {
                            return admHandler.changeStoreType( tInfo.getSubtreeName(), tInfo.getSource(), 
                                                               tInfo.isVerboseMode(), tInfo.isRecursive() );
                        }
                        
                        if ( tInfo.getAction().equalsIgnoreCase("naming") ) {
                            return admHandler.setNamingFlag( tInfo.getSubtreeName(), tInfo.getSource(), 
                                                             tInfo.isVerboseMode(), tInfo.isRecursive() );
                        }
                    }
                    
                    return sis.getInputStream();
                }
       
                if ( blockName.indexOf("/") != -1 && !blockName.trim().endsWith("/") ) {
                    int tID = handler.getTreeID( blockName.substring(0, blockName.lastIndexOf("/") ) );
                    
                    if ( tID != -1 && 
                         ( handler.getNaming( tID ).equalsIgnoreCase("Manual") ||
                           naming.equalsIgnoreCase("Manual") ) ) {
                        docName   = blockName.substring(blockName.lastIndexOf("/") + 1);
                        blockName = blockName.substring(0, blockName.lastIndexOf("/") ); 
                    }
                }

                int typeID = handler.getTreeID( blockName );
                int  docID = -1;
                
                if ( typeID != -1 ) {
                    String _mode = handler.getMode(typeID);
                    if ( _mode != null ) mode = _mode;
                }
		
                if ( typeID != -1 ) {
                    String _naming = handler.getNaming(typeID);
                    if ( _naming != null ) naming = _naming;
                }

                if ( action.equalsIgnoreCase("delete") ) {
                    if ( handler.hasTreeRights("read", blockName, false) &&
                         handler.hasTreeRights("write", blockName, false) ) {
                        return executeDelete( blockName, action, docName );
                    }
                    else {
                        result = LogManager.getReference().getMessage("delete_failed") + " '" + blockName;
                        if ( docName != null ) result += "/" + docName;
                        result += "' ";
                        if ( handler.getTreeID( blockName ) > -2 && handler.getDocumentID( blockName ) == -1 )
                            sis.write( getResponse("error", 0, 
                                                   result + LogManager.getReference().getMessage("not_document") ) );
                        else
                            sis.write( getResponse("error", 0 , 
                                                   result + LogManager.getReference().getMessage("permission_denied") ) );
                        return sis.getInputStream();
                    }
                }
                
                if ( action.equalsIgnoreCase("update") ) {

                    if ( docName != null ) {
                        int _dID = handler.getDocumentID( typeID, docName );
                        if ( _dID != -1 ) blockName += "/" + _dID ;
                    }

                    if ( handler.hasTreeRights("read", blockName, false) &&
                         handler.hasDocumentRights("read", blockName) &&
                         handler.hasDocumentRights("write", blockName) 
                         ) {
                        if ( (docID = updateXML( blockName )) == -1 ) { 
                            result = LogManager.getReference().getMessage("update_failed") + " '" + blockName;
                            if ( docName != null ) {
                                if ( !result.endsWith("/") ) result += "/";
                                result += docName;
                            } result += "' ";
                            sis.write( getResponse("error", 0, 
                                                   result + LogManager.getReference().getMessage("not_document") ) );
                            return sis.getInputStream();
                        }
                        isUpdate = true; 
                        blockName = blockName.substring(0, blockName.lastIndexOf("/")); 
                        action = "write";
                    }
                    else {
                        result = LogManager.getReference().getMessage("update_failed") + " '" + blockName;
                        if ( docName != null ) {
                            if ( !result.endsWith("/") ) result += "/";
                            result += docName;
                        } result += "' ";
   			
                        if ( handler.getTreeID( blockName ) > -2 && handler.getDocumentID( blockName ) == -1 )
                            sis.write( getResponse("error", 0, 
                                                   result + LogManager.getReference().getMessage("not_document") ) );
                        else
                            sis.write( getResponse("error", 0, 
                                                   result + LogManager.getReference().getMessage("permission_denied") ) );
                        return sis.getInputStream();
                    }
                }

                if ( action.equalsIgnoreCase("write") ) {
                    if ( !isUpdate && handler.getTreeID( blockName ) != -1 && 
                         ( handler.getDocumentID( blockName ) != -1 || 
                           ( docName != null && handler.getDocumentID(typeID, docName ) != -1 ) ) ) {
                        result = LogManager.getReference().getMessage("create_doc_failed") + " '" + blockName;
                        if ( docName != null ) {
                            if ( !result.endsWith("/") ) result += "/";
                            result += docName;
                        } result += "' ";
                        sis.write( getResponse("error", 0, result + LogManager.getReference().getMessage("already_used") ) );
                        return sis.getInputStream();
                    }
			 
                    if ( !handler.hasTreeRights("write", blockName, true) ) {
                        result = LogManager.getReference().getMessage("write_failed") + " '" + blockName + "' ";
                        sis.write( getResponse("error", 0, 
                                               result + LogManager.getReference().getMessage("permission_denied") ) );
                        return sis.getInputStream();
                    }
                    
                    int      length = getInputStreamLength(istream);
                    String   error  = writeDocument( blockName, docName, istream , mode , docID , isUpdate, action, info, true);
		    
                    if ( error != null ) {
                        try {
                            istream.close();
                        }	
                        catch( Exception e ) {
                            LogManager.getReference().message("error", "client_stream_not_closed", e.getMessage() );    
                        }
                        sis.write( getResponse("error", 0, error ) );
                        return sis.getInputStream();
                    }
                    
                    ( new LexiconManager() ).runLexiconManager( false );

                    String message = action + " " + blockName + "(" + length + " octets)";
                    LogManager.getReference().message("info", "store_successfull", message );
                    result = action + " action has been executed.";
                }
                else {
                    result = LogManager.getReference().getMessage("unknown_command") + action;
                    sis.write( getResponse("error", 0, result ) );
                    return sis.getInputStream();
                }
            }
        }
        else {
            result = LogManager.getReference().getMessage("sepengine_con_down") + " " + info.getName();
            LogManager.getReference().message("error", "sepengine_con_down", " " + info.getName() );
        }
        
        sis.write( getResponse("info", 0, result ) );
        return sis.getInputStream();
    }

    
    public String getHeader() {
        return header.toString();
    }


    public String getDTDName( InputStream is ) {
        String tag;     
        
        header.delete( 0, header.length() );
        
        do {
            tag = _nextToken( is );
	    if ( tag == null || !tag.trim().startsWith("<") ) return null;
	    tag = tag.trim();
            header.append( tag + "\n" );
        } while ( tag.startsWith("<!") || tag.startsWith("<?") || tag.length() < 1);
        tag = tag.replace('>', ' ').replace('<', ' ').trim();
        tag = tag.substring(0, tag.indexOf(' ') != -1 ? tag.indexOf(' ') : tag.length() );
    
        return tag;
    }

    
    public static void printIS( InputStream is ) {
        try {
            BufferedReader reader = new BufferedReader( new InputStreamReader( is ) );
            String           line;
            while ( ( line = reader.readLine() ) != null ) System.out.println( line );
            reader.close();
        }
        catch ( Exception e ) {} 
    }
    
    
    public String writeDocument( String blockName, String docName, InputStream xml , String mode, 
                                 int dID, boolean isUpdate, String action, RDBInfo info , boolean isStore) {
        String  dtdName = null;
        int         tID = -1;
        if ( !mode.toLowerCase().startsWith("free") ) {
            dtdName = getDTDName( xml );
            if ( dtdName == null ) 
                return "Write action failed." + LogManager.getReference().getMessage("not_xml");                    
            xml = new SepInputStream( xml, getHeader(), "");
        }
        
        if ( ( tID = handler._treeID( blockName ) ) != -1 && ( dID != -1 || ( dID = handler._messageID( tID ) ) != -1 ) ) {
            boolean modeFound = false;
    
            if ( mode.equalsIgnoreCase("flexible") ) {
                modeFound = true;
                if ( !storeDocumentInFlexibleMode(xml, blockName, docName, dtdName, tID, dID, isUpdate, info.getDTDDirInfo() ) )
                    return delDocument(tID, dID, action) + errorMsg;
            }           
    
            if ( mode.equalsIgnoreCase("strict") ) {
                modeFound = true;    
                if ( !storeDocumentInStrictMode(xml, blockName, docName, dtdName, tID, dID, isUpdate) )
                    return delDocument(tID, dID, action) + errorMsg;
            }
                        
            if ( mode.equalsIgnoreCase("free_tag") ) {
                modeFound = true;
                if ( !storeDocumentInFreeMode(xml, docName, tID, dID, isUpdate, true, isStore) )
                    return delDocument(tID, dID, action) + errorMsg;
            }
                        
            if ( mode.equalsIgnoreCase("free") ) {
                modeFound = true;
                if ( !storeDocumentInFreeMode(xml, docName, tID, dID, isUpdate, false, isStore) )
                    return delDocument(tID, dID, action) + errorMsg;
            }
        
            if ( !modeFound ) return LogManager.getReference().getMessage("unknown_mode");
        }    
        else {
            LogManager.getReference().message("error", "conf.db_error" , blockName);
            return LogManager.getReference().getMessage("doc_write_failed");
        }
        
        return null;
    }           

    
    public String normalize( String str ) {
        return str;
    }


    public String removeIgnorableKeys ( String str, Vector keys ) {
        StringBuffer sb = new StringBuffer( str );

        for ( Enumeration e = keys.elements(); e.hasMoreElements(); ) {
            String key = ( String )e.nextElement();
            int    poz = -1;
                                     
            while ( ( poz = sb.toString().indexOf( key ) ) != -1 )
                sb = sb.replace ( poz, poz + key.length(), " " );
        }

        return sb.toString();
    }

                                                                                                                       
    public String removeSpecialIgnorableKeys( String str, Vector keys ) {
        StringBuffer sb = new StringBuffer ( str );
        
        for ( Enumeration e = keys.elements(); e.hasMoreElements(); ) {
            String key = ( String )e.nextElement();
            int    poz = -1;    
            
            if ( sb.toString().startsWith( key ) ) sb.setCharAt(0, ' ');
            if ( sb.toString().endsWith( key ) ) sb.setCharAt(sb.length() -1 , ' ');                                                                                                                                       
            while ( ( poz = sb.toString().indexOf( " " + key ) ) != -1 )
                sb = sb.replace ( poz, poz + key.length() + 1 , " " );
            
            while ( ( poz = sb.toString().indexOf( key + " " ) ) != -1 ) 
                sb = sb.replace ( poz, poz + key.length() + 1, " " );
                
        }
        
        return sb.toString();
    }

    
    public InputStream getValidBlock (InputStream reader) {
        StringInputStream sis = new StringInputStream();
        String        rootTag = "";
        
        do {
            rootTag = _nextToken(reader);
            sis.write( rootTag );
        } while ( rootTag.trim().startsWith("<!") || 
                  rootTag.trim().startsWith("<?") || 
                  rootTag.trim().length() < 1);
            
        do {
            rootTag = _nextToken( reader );
            if ( rootTag == null ) break;
            
            if ( rootTag.trim().startsWith("<?") ) continue;
            if ( rootTag.trim().startsWith("<!") &&
                 !rootTag.toUpperCase().trim().startsWith("<![CDATA[") && 
                 !rootTag.trim().startsWith("<!--") ) continue;
                
            sis.write( rootTag );
                     
        } while ( true );
        
        return sis.getInputStream();
    }


    public String getFileName ( File f ) {
        String fName = null;
        
        if ( !f.canRead() ) return null; 
	
        for ( int i = 0; i < f.listFiles().length; i++ )
            if ( f.listFiles()[i].isFile() ) { 
                fName =  f.listFiles()[i].toString();
                fName = fName.substring( fName.lastIndexOf( File.separatorChar) + 1 );
                if ( fName.indexOf(".") != -1 ) fName = fName.substring(0, fName.indexOf(".") );
            }
        
        return fName;
    }
    

    public boolean createDTDFromXML(String blockName, String dName, String dtdName, InputStream xml, 
                                    int tID, int dID, DTDDirInfo info, boolean store) {
        String   myDTD = getDTDURI(blockName, dtdName, info);
	
        if ( !createDefaultDTD( myDTD, dtdName ) ) return false;
        try {
            new XMLAnalyzer( getBlock(dName, xml, dtdName , tID, dID, store , null ), blockName, myDTD, false );
        } 
        catch ( Exception e ) {
            errorMsg = e.getMessage();
            return false;
        }
        return true;
    }


    public boolean validateXML(String bName, String dName, String dtdName, InputStream xml, int tID, int dID, boolean store) {
        String myDTD = getDTDURI ( bName, dtdName, rdbInfo.getDTDDirInfo() );
        URL      url = null;
	
        try {
            url = new File( myDTD ).toURL();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "url_create_failed", " '" + myDTD + "'." + e.getMessage() );
            return false;
        }

        try {
            new XMLAnalyzer( getBlock(dName, xml, dtdName, tID, dID, store, url.toString() ), bName, myDTD, true );
        }
        catch ( Exception e ) {
            errorMsg = "\n" + LogManager.getReference().getMessage("url_found_failed") + " '" + myDTD + "' or \n" + e.getMessage();
            return false;
        }
        return true;
    }
    

    private String delDocument(int tID, int dID, String action){
        handler.deleteDocument(tID, dID, true);
        return action + " action failed.";
    }


    private String validBlock (InputStream is) {
        String rootTag = "", etag;
        
        do {
            rootTag = _nextToken(is).trim();
        } while ( rootTag.startsWith("<!") || rootTag.startsWith("<?") || rootTag.length() < 1 );

        StringBuffer sb = new StringBuffer(rootTag);
        String   endTag = rootTag.replace('<', ' ').replace('>', ' ').replace('/', ' ').trim();
        if ( endTag.indexOf(" ") != -1 ) endTag = endTag.substring(0, endTag.indexOf(" "));
        
        do {
            rootTag = _nextToken(is);
            if ( rootTag == null ) break;
            etag = rootTag.replace('<', ' ').replace('>', ' ').replace('/', ' ').trim();
            sb.append( rootTag );
            if ( etag.equalsIgnoreCase(endTag) ) break;
        } while ( true );
        
        return sb.toString();
    }

    
    private InputStream getBlock(String dName, InputStream reader, String endTag, int tID, int dID, boolean store, String dtdUrl) {
        StringInputStream         sis = new StringInputStream();
        StringBuffer               sb = new StringBuffer();
        String                rootTag = null;
        boolean       foundXMLVersion = false;
        boolean foundDTDSpecification = false;
	
        blockNr = 0;
	
        try {
        
            do {
                if ( rootTag != null && !foundDTDSpecification ) {
                    sb.append( rootTag );
                    sis.write( rootTag );
                    
                }
                foundDTDSpecification = false;
                rootTag = _nextToken( reader );
                String rtag = rootTag.replace('<', ' ').replace('?', ' ').replace('!', ' ' ).trim().toUpperCase();
                if ( rootTag.trim().startsWith("<?") && rtag.startsWith("XML") ) 
                    foundXMLVersion = true;
                if ( rootTag.trim().startsWith("<!") && rtag.startsWith("DOCTYPE") ) 
                    foundDTDSpecification = true;
            } while ( rootTag.trim().startsWith("<!") || 
                      rootTag.trim().startsWith("<?") || 
                      rootTag.trim().length() < 1);
            
            if ( !foundXMLVersion && dtdUrl != null ) {
                sb.append( "<?xml version='1.0'?>\n" );
                sis.write( "<?xml version='1.0'?>\n" );
	    }
	    
	    if ( dtdUrl != null ) {
    		sb.append( "<!DOCTYPE " + endTag + " SYSTEM '" + dtdUrl + "'>\n" );
        sis.write( "<!DOCTYPE " + endTag + " SYSTEM '" + dtdUrl + "'>\n" );
	    }
	
	    sb.append( rootTag );
	    sis.write( rootTag );
        
	    if ( store && sb.length() > BLOCK_SIZE )
                sb = sb.delete(0, writeBuffer(dName, sb, tID, dID, false) );
            
            boolean first = true;
            do {
                rootTag = _nextToken( reader );
                if ( rootTag == null ) break;
                if ( first ) rootTag = rootTag.substring(1);
		
                sb.append( rootTag );first = false;

                if ( store && sb.length() > BLOCK_SIZE )
                    sb = sb.delete(0, writeBuffer(dName, sb, tID, dID, false) );
                
                String etag = rootTag.replace('<', ' ').replace('>', ' ').replace('/', ' ').trim();
                if ( etag.equalsIgnoreCase(endTag) ) {
                    sis.write(rootTag);
                    break;
                }
        
                if ( rootTag.trim().startsWith("<?") ) continue;
                if ( rootTag.trim().startsWith("<!") &&
                     !rootTag.toUpperCase().trim().startsWith("<![CDATA[") && 
                     !rootTag.trim().startsWith("<!--") ) continue;
                
                sis.write( rootTag );
                     
            } while ( true );
            
            if ( store ) write2Rep(dName, sb, 0, sb.length(), tID, dID, true, false);
        } 
        catch ( Exception _exe ) {
            LogManager.getReference().message("error", "inputstream_error", _exe.getMessage() );
            return null;
        }
        
        return sis.getInputStream();
    }

    private SepReturnTable retTable(InputStream is, String uname) {
        SepReturnTable ret = new SepReturnTable();
        try {
            sepLexer     lexer = new sepLexer( new DataInputStream(is) );
            sepParser   parser = new sepParser(lexer);
            ret = parser.sepStart(rdbInfo, uname, this);
        }
        catch ( Exception e ) {
            e.printStackTrace();
            ret.error = LogManager.getReference().getMessage("sep_parser_excep") +
                ":\n" + e.getMessage() + "\n";
        }
        return ret;
    }


    private InputStream executeFetch ( String sql ,boolean scan ) {
        StringInputStream sis = new StringInputStream(); 
        String result = "";
        if ( sql.trim().length() > 0 ) {
            if ( sql.startsWith("ERROR") ) result = sql.substring( 5 );
            else 
                if ( sql.startsWith("_admin.") )
                    return admHandler.execute( sql.substring(0, sql.indexOf(":") ) ,
                                               sql.substring(sql.indexOf(":")+1) );
                else 
                    return handler.execute(sql, scan);
        }
        else {
            result = LogManager.getReference().getMessage("sep_parser_unrep");
        }
        sis.write( getResponse("error", 0, result ) ); 
        return sis.getInputStream();
    }

    
    private InputStream executeDelete( String blockName, String action, String docName ) {
        StringInputStream sis = new StringInputStream(); 
        String result = null;

        if ( !deleteXML( blockName, docName ) ) { 
            result = action + " action failed.\n" + errorMsg;
            sis.write( getResponse("error", 0, result) );
        }
        else {
            LogManager.getReference().message("info", "store_successfull", action + " " + blockName + "." );
            result = action + " action has been executed.";
            sis.write( getResponse("info", 0, result) );
            (new LexiconManager()).runLexiconManager( false );
        }
        
        return sis.getInputStream();
    }

    
    private int getInputStreamLength(InputStream is) {
        int length = 0;

        try {
            length = is.available();
        } catch (IOException e){}
        
        return length;
    }
    

    private boolean storeDocumentInFlexibleMode(InputStream xml, String bName, String dName, String dtdName, 
                                                int typeID, int docID, boolean isUpdate, DTDDirInfo info) {
        if ( !createDTDFromXML(bName, dName, dtdName, xml, typeID, docID, info, true) ) return false;
        if ( isUpdate && !handler.markDocument(typeID, docID, "U") ) return false;
        return true;
    }

    private boolean storeDocumentInStrictMode(InputStream xml, String bName, String dName, String dtdName, 
                                              int typeID, int docID, boolean isUpdate) {
        if ( !validateXML(bName, dName, dtdName, xml, typeID, docID, true) ) return false;
        if ( isUpdate && !handler.markDocument(typeID, docID, "U") ) return false;
        return true;
    }

    
    private boolean storeDocumentInFreeMode(InputStream xml, String docName, int tID, int dID, boolean isUpdate, boolean withTag, boolean isStore) {
        try {
            BufferedReader reader = new BufferedReader( new InputStreamReader( xml ) );
            char[]         buffer = new char[ BLOCK_SIZE ];
            int              size = 0;
            StringBuffer       sb = new StringBuffer();
            
            blockNr = 0;
	    
            while ( true ) {
                if ( ( size = reader.read( buffer, 0, BLOCK_SIZE ) ) == -1 ) break;
                sb.append( new String( buffer, 0, size ) );
                sb = sb.delete(0, writeBuffer(docName, sb, tID, dID, true) );
            }
            write2Rep(docName, sb, 0, sb.length(), tID, dID, isStore, true);
        }
        catch ( IOException ioe ) {
            LogManager.getReference().message("error", "inputstream_error", ioe.getMessage() );
            return false;
        }
        
        if ( isUpdate && !handler.markDocument(tID, dID, "U") ) return false;
        return true;
    }
    

    private String separatorNeeded( String path ) {
        if( !path.trim().endsWith( File.separator ) )
            path += File.separator;
        return path;
    }


    private String getSEPCommand( InputStream is ) {
        StringBuffer sb = new StringBuffer();
        int           c = -1;
	
        try {
            while ( ( c = is.read() ) != -1 && ( c != '>' ) ) {
                if ( c != '<' ) sb.append( (char)c );
                if ( sb.toString().trim().length() > 0 && c == ' ' ) break;
            }	
		
        }
        catch ( Exception _ex ) {
            LogManager.getReference().message("error", "sep_command_error", _ex.getMessage() );
        }
        
        return sb.toString().trim();
    } 


    private String getDTDURI(String blockName, String dtdName, DTDDirInfo info) {
        String DTDDir   = File.separator;
        File   file     = null;
                             
        if ( dtdName == null || blockName == null ) return null;
                                     
        if ( info.getDTDPath() != null )
            DTDDir = info.getDTDPath().replace('/', File.separatorChar);
                                 
        try {
            String url = System.getProperty("java.ext.dirs");
            int   poz;
         
            if ( (poz = url.indexOf(":")) != -1 || (poz = url.indexOf(";")) !=-1) url = url.substring(0, poz);
            if ( !DTDDir.startsWith(File.separator) ) DTDDir = url + File.separator + ".." + File.separator + DTDDir;
                                                                                 
            file = new File( DTDDir );
            //if DTD directory not exists will be automat created
            if ( !file.exists() ) file.mkdir();
            
            return separatorNeeded( separatorNeeded( DTDDir ) + 
                                    blockName.replace('.', File.separatorChar)) +
                dtdName + ".dtd";
                                
        }
        catch ( Exception _cnfe ) {
            errorMsg = LogManager.getReference().getMessage("create_dtd_failed");
            LogManager.getReference().message("error", "conf.file_missing" , _cnfe.getMessage());
            return null;
        }
    }


    private boolean createDefaultDTD(String path, String name) {
        if ( path == null || name == null ) return false;
        File f = new File ( path );
	
        if ( !f.exists() ) {
            f = new File ( path.substring( 0, path.lastIndexOf( File.separatorChar ) ) );
            if ( !f.exists() ) f.mkdirs();
	    
            String fName = getFileName(f);
            
            if ( fName == null ) {
                try {
                    DataOutputStream dout = new DataOutputStream(new FileOutputStream( path ));
                    dout.writeBytes("<!ELEMENT " + name + " {SPECIAL}>");
                }
                catch( Exception e ) {
                    errorMsg = "\n" + LogManager.getReference().getMessage("create_dtd_failed") + e.getMessage();
                    LogManager.getReference().message("error", "file_create_fail",  e.getMessage() );
                    return false;
                }
            }
            else {
                errorMsg = LogManager.getReference().getMessage("invalid_root") + fName;
                return false;
            }
        }
        return true;
    }

    
    private boolean deleteXML( String blockName, String docName ) {
        int tID = handler.getTreeID( blockName );
        int dID = handler.getDocumentID("");
                         
        if ( docName != null ) dID = handler.getDocumentID( tID, docName );

        if ( tID == -1 || dID == -1 ) {
            errorMsg = LogManager.getReference().getMessage("incorect_blockname") + blockName;
            return false;
        }
        if ( handler.markDocument(tID, dID, "D") ) return true;
        return false;
    } 


    private int updateXML( String blockName ) {
        int tID = handler.getTreeID( blockName );
        int dID = handler.getDocumentID("");
                         
        if ( tID == -1 || dID == -1 ) {
            errorMsg = LogManager.getReference().getMessage("incorect_blockname") + blockName;
            return -1;
        }
        if ( handler.deleteDocument(tID, dID, false) ) return dID;
        return -1;
    } 

    
    private String _nextToken(InputStream is) {
        StringBuffer result = new StringBuffer();
        try {
            int ch = 0;
            
            while( (ch = is.read() ) != -1 ) {
                result.append((char)ch);
                if ( ch == '>') {
                    if ( result.toString().trim().startsWith("<!--") ) {
                        if ( result.toString().trim().endsWith("-->") )  return result.toString();
                        else continue;
                    }
                    else {
                        if (result.toString().trim().startsWith("<!"))
                            if ( result.toString().indexOf("[") == -1 ) return result.toString();
                            else 
                                if ( result.toString().indexOf("[") != -1 &&
                                     result.toString().indexOf("]") != -1) return result.toString();
                                else continue;
                    }
                    return result.toString();
                } 
            }
            return null;
        }
        catch ( IOException ioex ) {
            LogManager.getReference().message("error", "inputstream_error", " " + ioex.getMessage() );
            return null;
        }
    }
    

    private int writeBuffer( String dName, StringBuffer block, int tID, int dID, boolean freeMode) {
        int offset = 0;
        
        while ( ( block.length() - offset) > BLOCK_SIZE ) {
            offset = write2Rep(dName, block, offset, offset + BLOCK_SIZE, tID, dID, false, freeMode);
        }
        
        return offset;
    }
    

    private int write2Rep(String dName, StringBuffer sb, int start, int end, int tID, int dID, boolean last, boolean freeMode) {
        if ( end > sb.length() ) end = sb.length();
        if ( !last && ( end = sb.toString().lastIndexOf(' ', end) ) < start ) end = sb.length();
        
        if ( start > end ) return start;
        
        String str = sb.substring(start, end);
        int    poz = -1;
        
        if ( freeMode && last && ( poz = str.lastIndexOf("</") ) != -1 ) {
            str = ("+" + str).trim().substring(1);
            if ( str.indexOf(">", poz) == str.length() - 1 ) {
                str = ("+" + str.substring(0, poz)).trim().substring(1); 
                if ( ( poz = str.lastIndexOf("</") ) != -1 ) {
                    if ( str.indexOf(">", poz) == str.length() - 1 ) {
                        str = ("+" + str.substring(0, poz)).trim().substring(1);
                        if ( ( poz = str.lastIndexOf("</") ) != -1 ) {
                            if ( str.indexOf(">", poz) == str.length() - 1 ) {
                                str = str.substring(0, poz);
                            }
                        } 
                    }
                }
            }
            end = poz;
        }
        handler.writeToRepository ( dName, str.trim(), ++blockNr, tID, dID );

        return end;
    }

    
    private String getValue( String token ) {
        String quote = null;
        if ( token.indexOf("'")  != -1 ) quote = "'";
        if ( token.indexOf("\"") != -1 ) quote = "\"";
        if ( quote == null ) return null;
        int start = token.indexOf(quote)+1;
        int   end = token.indexOf(quote, start);
        if ( start == -1 || end == -1 ) 
            return null;
        else
            return token.substring(start, end);
    }
}
