/* $COPYRIGHT$
 * $Id: SepADMDBHandler.java,v 1.42 2002/05/20 17:41:03 crow Exp $
 *
 * Date         Author           Changes
 * JUL 15 2001  Szabo Csaba      Created
 */
 
package org.media.datastore.sepengine.util;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.sql.ResultSet;

import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;

import java.io.InputStream;
import org.beepcore.beep.profile.sasl.otp.*;

import org.media.datastore.sepengine.LogManager;
import org.media.datastore.sepengine.*;
import org.media.datastore.sepengine.util.StringInputStream;

/**
 * This class .....
 * @author <a href="mailto:crow@nolimits.ro">Szabo Csaba</a>
 * @version $Revision: 1.42 $ $Date: 2001/05/21 22:30:30
 * @see java.lang.String
 * @see org.media.datastore.sepengine.RDBInfo
 * @see org.media.datastore.sepengine.ChownInfo
 * @see org.media.datastore.sepengine.ChmodInfo
 * @see org.media.datastore.sepengine.GroupInfo
 * @see org.media.datastore.sepengine.UserInfo
 * @see org.media.datastore.sepengine.TreeInfo
 * @see org.media.datastore.sepengine.util.StringInputStream
 * @see org.media.datastore.sepengine.util.SepDBHandler
 */

public class SepADMDBHandler {
    private static final String TABLE_PREFIX = "ds";
    
    private static int MAX_USER_SIZE  = 0; 
    private static int MAX_GROUP_SIZE = 0;
    
    private String       errorMsg;
    private RDBInfo      rdbInfo;
    private String       rdbName;
    private String       uName;
    private int          user_id;
    private int          group_id;
    private SepDBHandler handler;
    private SepInterpreter si;
    private Connection   con;
    

    public SepADMDBHandler( RDBInfo info, String uname, SepInterpreter si ) {
        this.si  = si;
        handler  = new SepDBHandler(info, uname, si);
        errorMsg = null;
        rdbInfo  = info;
        rdbName  = info.getName();
        uName    = uname;
        if ( uname.equals("root") ) {
            user_id  = -1;
            group_id = -1;
        }
        else {
            if ( connect() ) {
                String sql = "SELECT user_id FROM " + TABLE_PREFIX + "users WHERE user_name='" + uname + "'";
                try {
                    Statement stmt = con.createStatement();
                    ResultSet   rs = stmt.executeQuery( sql );
                    if ( rs.next()  ) {
                        user_id = rs.getInt( "user_id" );
                        group_id = getGroupID( uname );
                    }
                    rs.close();stmt.close();
                }
                catch ( Exception e ) {}
                disconnect();
            }
        }
    }


    public String getErrorMSG() {
        return errorMsg;
    }
        

    public boolean hasRights() {
        return user_id == -1;
    }
    

    public boolean removeTree ( String tName, boolean forced ) {
        int tID = handler.getTreeID( tName );
        int dID = handler.getDocumentID("");
        
        if ( tID == -1 ) {
            errorMsg = "Subtree " + LogManager.getReference().getMessage("not_exist");
            return false;
        }
            
        if ( dID != -1 ) {
            if ( handler.hasTreeRights("read", tName, false) && handler.markDocument(tID, dID, "D") ) {
                (new LexiconManager()).runLexiconManager( false );      
                return true;
            }
            else {
                errorMsg = LogManager.getReference().getMessage("permission_denied") + "!";
                return false;
            }
        }
        else {
            if ( !forced ) {
                if( !handler.isEmptyTree( tID ) ) {
                    errorMsg = LogManager.getReference().getMessage("subtree_not_empty");
                    return false;
                }
                if ( !hasRights() && user_id != handler.getOwnerID(tID) ) {
                    errorMsg = LogManager.getReference().getMessage("permission_denied") + "!";
                    return false;
                }
                errorMsg = "";
                return handler.deleteTree(tID);
            }
            else {
                errorMsg = "";
                boolean error = true;
                if ( connect() ) {
                    error = removeTreeRecursively(tName);
                    disconnect();
                }
                if ( !error ) error = handler.deleteTree(tID);
                return error;
            }
        }
    }
   

    public InputStream listTree( String tName, boolean isLong, boolean istree) {
        StringInputStream sis = new StringInputStream();
        int tID = handler.getTreeID( tName );
        int dID = handler.getDocumentID("");
        
        if ( connect() ) {
            if ( tID == -1 || !handler.hasTreeRights("read", tName, false) ) 
                sis.write( si.getResponse("error", 0 ,"'" + tName + "' " + LogManager.getReference().getMessage("doc_not_found") ) );
            else         
                if ( dID != -1 ) {
                    Vector docs = longFormatDocs( "doc_id=" + dID + " AND tree_id=" + tID );
                    
                    if ( docs != null ) {
                        if ( isLong ) 
                            sis.write ( si.getResponse("info", 0, ( (LongFormat)docs.elementAt(0) ).toString() + "\n" ) ); 
                        else 
                            sis.write ( si.getResponse("info", 0, tName ) );
                    }
                    else {
                        sis.write ( si.getResponse("info", 0, 
							       "'" + tName + "' " + 
							       LogManager.getReference().getMessage("subtree_empty") ) );
                    }
                }
                else {
                    if ( istree ) {
                        Vector trees = longFormatTrees( "tree_id=" + tID );
                        
                        if ( trees != null ) {
                            if ( isLong ) 
                                sis.write ( si.getResponse("info", 0, ((LongFormat)trees.elementAt(0)).toString() + "\n" ) ); 
                            else 
                                sis.write ( si.getResponse("info", 0, tName ) ); 
                        }
                        else {
                            sis.write ( si.getResponse("info", 0, "'" + tName + "' " +
                                                       LogManager.getReference().getMessage("subtree_empty") ) );
                        }
                    }
                    else {
                        Vector trees = longFormatTrees( "tree_parent=" + tID );
                        Vector docs  = longFormatDocs ( "tree_id=" + tID );
                        
                        sis.write("<response reqno='" + si.getRequestNr() + "'>\n" +
                                  "<additional>\n" +
                                  "<info code=\"0\">\n");
                                  
                        if ( trees != null ) {
                            for ( Enumeration e = trees.elements(); e.hasMoreElements(); ) {
                                LongFormat lf = (LongFormat)e.nextElement();
                                if ( isLong ) 
                                    sis.write ( lf.toString() + "\n"); 
                                else 
                                    sis.write ( lf.getHomeDir() + "\n");
                            }
                        }
                        if ( docs != null ) {
                            for ( Enumeration e = docs.elements(); e.hasMoreElements(); ) {
                                LongFormat lf = (LongFormat)e.nextElement();
                                if ( isLong ) 
                                    sis.write ( lf.toString() + "\n"); 
                                else 
                                    sis.write ( lf.getHomeDir() + "\n");
                            }
                        }
                        
                        sis.write( "</info>\n" +
                                   "</additional>\n" +
                                   "</response>\n" );
                
                        if ( docs == null && trees == null ) {
                            sis.flush();
                            sis.write ( si.getResponse("info", 0, "'" + tName + "' " + 
                                                       LogManager.getReference().getMessage("subtree_empty") ) );
                        }
                    }
                }
            disconnect();
        }
        
        return sis.getInputStream(); 
    }


    public InputStream copyTree( String src, String dest, boolean verbose, boolean recursive ) {
        StringInputStream sis = new StringInputStream();
        int  srcTreeID = handler.getTreeID( src );
        int   srcDocID = handler.getDocumentID("");
        int destTreeID = handler.getTreeID( dest );
        int  destDocID = handler.getDocumentID(""); 
        
        if ( connect() ) {
            if ( srcTreeID == -1 )
                sis.write ( si.getResponse("error", 0, 
                                           "'" + src + "' " + LogManager.getReference().getMessage("subtree_not_found") ) );
            else
                if ( destTreeID == -1 || destDocID != -1 )
                    sis.write ( si.getResponse("error", 0, "'" + dest + "' " + 
                                               LogManager.getReference().getMessage("subtree_not_found") ) );
                else 
                    if ( !handler.hasTreeRights("read", src, false ) )
                        sis.write( si.getResponse("error", 0,
                                                  LogManager.getReference().getMessage("src_read_denied") + 
                                                  " '" + src + "'.") );
                    else
                        if ( !handler.hasTreeRights("write", dest, false) ) 
                            sis.write( si.getResponse("error", 0,
                                                      LogManager.getReference().getMessage("dest_write_denied") + 
                                                      " '" + dest + "'.") );
                        else {
                            if ( srcDocID != -1 ) {
                                if ( handler.hasDocumentRights("read", src ) ) { 
                                    String error = copyDocument( dest, srcTreeID, srcDocID );
                                    if ( error == null ) { 
                                        if ( verbose ) 
                                            sis.write ( si.getResponse("info", 0, src + " -&gt; " + dest ) );
                                        else 
                                            sis.write ( si.getResponse("info", 0, "1 " + 
                                                                       LogManager.getReference().getMessage("file_copied") + 
                                                                       " '" + dest + "'.") );
                                    }
                                    else sis.write ( si.getResponse("error", 0, error ) );
                                }
                                else {
                                    sis.write( si.getResponse("error", 0,
                                                              LogManager.getReference().getMessage("doc_copy_failed") + " " + 
                                                              src + ": " + 
                                                              LogManager.getReference().getMessage("permission_denied") + 
                                                              "!" ) );
                                }
                            }
                            else {
                                sis = copyTree( sis, dest, src, srcTreeID , recursive , verbose, true );
                            }
                        }    
            disconnect();
        }
        
        ( new LexiconManager() ).runLexiconManager( false ); 
                
        return sis.getInputStream();
    }
    

    public InputStream changeStoreType( String subtree, String storeType, boolean verbose, boolean recursive ) {
        StringInputStream sis = new StringInputStream();
        int tID = handler.getTreeID( subtree );
        int dID = handler.getDocumentID("");
        
        if ( connect() ) {
            if ( tID == -1 || dID != -1 )
                sis.write ( si.getResponse("error", 0, "'" + subtree + "' " + 
                                           LogManager.getReference().getMessage("subtree_not_found") ) );
            else if ( !handler.hasTreeRights("read", subtree, false ) )
                sis.write( si.getResponse("error", 0,
                                          LogManager.getReference().getMessage("src_read_denied") + " '" + subtree + "'.") );
            else if ( !handler.hasTreeRights("write", subtree, false ) ) 
                sis.write( si.getResponse("error", 0,
                                          LogManager.getReference().getMessage("dest_write_denied") + " '" + subtree + "'.") );
            else sis = setStoreType( sis, subtree, tID , storeType, verbose, recursive , true);
            
            disconnect();
            
        }
	
        ( new LexiconManager() ).runLexiconManager( false ); 
        
        return sis.getInputStream();
    }


    public InputStream setNamingFlag( String subtree, String flag, boolean verbose, boolean recursive ) {
        StringInputStream sis = new StringInputStream();
        int tID = handler.getTreeID( subtree );
        int dID = handler.getDocumentID("");
        
        if ( connect() ) {
            if ( tID == -1 || dID != -1 )
                sis.write ( si.getResponse("error", 0, "'" + subtree + "' " + 
                                           LogManager.getReference().getMessage("subtree_not_found") ) );
            else if ( !handler.hasTreeRights("read", subtree, false ) )
                sis.write( si.getResponse("error", 0,
						      LogManager.getReference().getMessage("src_read_denied") + " '" + subtree + "'.") );
            else if ( !handler.hasTreeRights("write", subtree, false ) ) 
                sis.write( si.getResponse("error", 0,
                                          LogManager.getReference().getMessage("dest_write_denied") + " '" + subtree + "'.") );
            else sis = setNamingFlag( sis, subtree, tID , flag, verbose, recursive , true);
            
            disconnect();
            
        }
        
        return sis.getInputStream();
    }

    
    public InputStream chmod ( ChmodInfo ci ) {
        StringInputStream sis = new StringInputStream();
        int      tID = handler.getTreeID( ci.getDoc() );
        int      dID = handler.getDocumentID("");
        
        if ( connect() ) { 
            if ( tID == -1 ) 
                sis.write( si.getResponse("error", 0, "'" + ci.getDoc() + "' " + 
                                          LogManager.getReference().getMessage("doc_not_found") ) ); 
            else 
                if ( dID != -1 ) {
                    String result = chmodForDocument(tID, dID, ci.getMode(), ci.isVerbose(), ci.getDoc() );
                    if ( result == null ) { 
                        sis.write( si.getResponse("error", 0, errorMsg ) );
                    }
                    else {
                        if ( result.trim().length() < 1 ) result = LogManager.getReference().getMessage("chmod_successfull");
                        sis.write( si.getResponse( "info", 0, result ) );
                    }
                }
                else {
                    String result = chmodForTree(tID, ci.getMode(), ci.isRecursive(), ci.isVerbose(), ci.getDoc() );
                    if ( result == null )
                        sis.write( si.getResponse("error", 0, errorMsg ) );
                    else
                        sis.write( si.getResponse("info", 0, result ) );
                }
            disconnect();
        }
        
        return sis.getInputStream();
    }

    
    public InputStream chown ( ChownInfo ci ) {
        StringInputStream sis = new StringInputStream();
        int      tID = handler.getTreeID( ci.getDoc() );
        int      dID = handler.getDocumentID("");
        int      uID = -2, gID = -2;
        String owner = ci.getOwner().trim();
        int      poz = owner.indexOf(".");
        String uname = null, gname = null;
        
        if ( connect() ) { 
            if ( poz == -1 ) {
                uID = getUserID( uname = owner );
            }
            else {
                if ( owner.endsWith(".") ) {
                    uID = getUserID( uname = owner.substring( 0, poz ) );
                    gID = getGroupID( gname = owner.substring( 0, poz ) );
                }
                else {
                    if ( owner.startsWith(".") ) {
                        gID = getGroupID( gname = owner.substring( poz + 1 ) );
                    }
                    else {
                        uID = getUserID( uname = owner.substring( 0, poz ) );
                        gID = getGroupID( gname =  owner.substring( poz + 1 ) );
                    }
                }
            }
            
            if ( uID == -1 )
                sis.write( si.getResponse("error", 0, "User '" + uname + "' " + 
                                          LogManager.getReference().getMessage("not_exist") ) );
            else 
                if ( gID == -1 )
                    sis.write( si.getResponse("error", 0, "Group '" + gname + "' " + 
                                              LogManager.getReference().getMessage("not_exist") ) );
                else
                    if ( tID == -1 ) 
                        sis.write( si.getResponse("error", 0, "'" + ci.getDoc() + "' " + 
                                                  LogManager.getReference().getMessage("doc_not_found") ) ); 
                    else 
                        if ( dID != -1 ) {
                            String result = chownForDocument(tID, dID, uID, gID, ci.isVerbose(), ci.getDoc() );
                            if ( result == null ) {
                                sis.write( si.getResponse("error", 0, errorMsg ) );
                            }
                            else {
                                if ( result.trim().length() < 1 ) result = LogManager.getReference().getMessage("chown_successfull");
                                sis.write( si.getResponse("info", 0, result ) );
                            }
                        }
                        else {
                            String result = chownForTree(tID, uID, gID, ci.isRecursive(), ci.isVerbose(), ci.getDoc() );
                            if ( result == null )
                                sis.write( si.getResponse("error", 0, errorMsg ) );
                            else
                                sis.write( si.getResponse("info", 0, result ) );
                        }
            disconnect();
        }
        
        return sis.getInputStream();
    }
        

    public boolean addGroup ( GroupInfo gi ) {
        if ( connect() ) {
            String gname = gi.getGroupName();
            int        gID = getGroupID( gname );
            
            if ( gID != -1 ) {
                errorMsg = "Group " + gname + " " + LogManager.getReference().getMessage("exist");
                return false;
            }
            try {
                addNewGroup ( gname );
            }
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("group_add_error");
                return false;
            }
            disconnect();
        }
        return true;
    }


    public boolean modGroup ( GroupInfo gi ) {
        if ( connect() ) {
            String   gname = gi.getGroupName();
            int        gID = getGroupID( gname );
            String newName = gi.getNGroupName();
            
            if ( gID == -1 ) {
                errorMsg = "Group " + gname + " " + LogManager.getReference().getMessage("not_exist");
                return false;
            }
            
            if (  getGroupID( newName ) != -1 ) {
                errorMsg = "Group " + newName + " " + LogManager.getReference().getMessage("exist");
                return false;
            } 
            
            try {
                updateGroup ( gID, newName );
            }
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("group_update_error");
                return false;
            }
            disconnect();
        }
        return true;
    }

    
    public boolean delGroup ( GroupInfo gi ) {
        if ( connect() ) {
            String gname = gi.getGroupName();
            int      gID = getGroupID( gname );

            if ( gID == -1 ) {
                errorMsg = "Group " + gname + " " + LogManager.getReference().getMessage("not_exist");
                return false;
            }
            
            try {
                if ( groupHasUsers( gID ) ) {
                    errorMsg = LogManager.getReference().getMessage("group_has_users");
                    return false;
                }
                deleteGroup ( gID );
            }
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("group_del_error");
                return false;
            }
            disconnect();
        }
        return true;
    }
    

    public Vector getUsers() {
        Vector result = new Vector();
        
        if ( connect() ) {
            try {
                Statement stmt = con.createStatement();
                ResultSet   rs = stmt.executeQuery( "SELECT * FROM " + TABLE_PREFIX + "users");
                while ( rs.next() ) {
                    UserInfo ui = new UserInfo();
                    ui.setUsername( rs.getString("user_name") );
                    ui.setPassword( rs.getString("user_passwd") );
                    result.addElement(ui);
                }
                rs.close();stmt.close();
            }
            catch ( Exception e ) {
                LogManager.getReference().message("error", "get_user_data_failed" , e.getMessage() );
            }
            disconnect();
        }
        
        return result;
    }


    public boolean addUser( UserInfo ui ) {
        int        hID = -1;
        String   uname = ui.getUserName();
        String   gname = ui.getInitialgroup();
        String    upwd = ui.getPassword() != null ? ui.getPassword() : ""; 
        Vector  groups = ui.getGroups();
        
        if ( connect() ) {
            if ( userExist( uname ) ) {
                errorMsg = "Username " + uname + " " + LogManager.getReference().getMessage("exist");
                return false;
            }
            if ( ( user_id = nextID("user_id") ) == -1 ) {
                errorMsg = LogManager.getReference().getMessage("user_add_error");
                return false;
            }
            
            try {
                group_id = addNewGroup( gname );
                handler.setUserID( user_id );
                handler.setGroupID( group_id );
                hID = ui.getHomedir() != null ? handler._treeID( ui.getHomedir() ) : -1;
                addUserToGroup( user_id, group_id );
                addNewUser( user_id, uname, si.preDigest (upwd), hID );
                MyOTPProfile.addUser( rdbName + "_" + uname, 
                                      si.getHashMethod(), si.getPasswdHash( si.preDigest (upwd) ) , 
                                      si.getSeed(), si.getSeq() );
                if ( groups != null ) {
                    for ( Enumeration en = groups.elements(); en.hasMoreElements(); ) {
                        gname = (String)en.nextElement();
                        group_id = addNewGroup ( gname );
                        addUserToGroup ( user_id, group_id );
                    }
                } 
            }
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("user_add_error");
                try {
                    removeUser ( user_id );
                } catch ( Exception _ex ){}
                return false;
            }
            disconnect();
        }
        
        return true;
    }
    

    public boolean modUser( UserInfo ui ) {
        String  lname = ui.getLoginName();
        String  uname = ui.getUserName();
        String   upwd = ui.getPassword();
        String  gname = ui.getInitialgroup();
        Vector groups = ui.getGroups();
        
        if ( connect() ) {
            if ( !userExist ( lname ) ) {
                errorMsg = "Username " + lname + " " + LogManager.getReference().getMessage("not_exist");
                return false;
            }
            
            try {
                int   uID = getUserID( lname );
                int   gID = -1; 
                
                if ( gname != null || groups != null ) deleteUserGroups ( uID );

                if ( groups != null ) {
                    int ngid = -1;
		    
                    for ( Enumeration en = groups.elements(); en.hasMoreElements(); ) {
                        String grname = (String)en.nextElement();
			
                        gID = addNewGroup( grname );
                        addUserToGroup ( uID, gID );
                        if ( ngid == -1 ) ngid = gID; 
                    }
                    con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET group_id=" + ngid + " WHERE user_id=" + uID).execute();
                    con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET group_id=" + ngid + " WHERE user_id=" + uID).execute();
                } 

                if ( gname != null ) { 
                    gID = addNewGroup( gname );
                    addUserToGroup( uID, gID );
                    con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET group_id=" + gID + " WHERE user_id=" + uID).execute();
                    con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET group_id=" + gID + " WHERE user_id=" + uID).execute();
                }
                else {
                    gID = getGroupID( uID );
                }
                    
                handler.setUserID( -2 );
                handler.setGroupID( gID );

                String   myPWD = null;
                if ( upwd != null ) myPWD = si.preDigest ( upwd );
    
                int        hID = ui.getHomedir() != null ? handler._treeID( ui.getHomedir() ) : -1;
                int      oldID = updateUser(lname, uname, myPWD , hID);
                String myfName = rdbName + "_";
	
                if ( uname != null ) myfName += uname; else myfName += lname;
                MyOTPProfile.removeUserDB( rdbName + "_" + lname );
                Statement stmt = con.createStatement();
                ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "users WHERE user_name='" + lname + "'" );
                if ( rs.next() ) upwd = rs.getString("user_passwd"); rs.close(); stmt.close();
                MyOTPProfile.addUser(myfName, si.getHashMethod(), si.getPasswdHash( upwd ), 
                                     si.getSeed(), si.getSeq() );
                
                if ( oldID > 0 && hID > 0 ) {
                    moveDocuments( oldID, hID );
                    handler.deleteTree( oldID );
                }
                
                con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET user_id=" + uID + " WHERE user_id=-2").execute();
                 
            }
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("user_update_error");
                return false;
            }
            disconnect();
        }
        return true;
    }

        
    public boolean delUser( UserInfo ui ) {
        String uname = ui.getUserName();
        int      uID = -1;
        
        if ( connect() ) {
            if ( !userExist( uname ) || ( uID = getUserID(uname) ) == -1 ) {
                errorMsg = "Username " + uname + " " + LogManager.getReference().getMessage("not_exist");
                return false;
            }
            try {
                removeUser( uID );
                MyOTPProfile.removeUserDB( rdbName + "_" + uname );
            } 
            catch ( Exception e ) {
                errorMsg = LogManager.getReference().getMessage("user_del_error");
                return false;
            }
            disconnect();
        }
        return true;
    }
    

    public boolean blockUser( UserInfo ui ) {
        boolean found = false;
        
        if ( connect() ) {
            String sql = "UPDATE " + TABLE_PREFIX + "users SET blocked='t' WHERE upper(blocked)='F' AND user_name!='root' AND user_name='" + 
                ui.getUserName() + "'";
            try {
                found = con.prepareStatement(sql).executeUpdate() > 0;
            }
            catch ( Exception e ) {
                found = false;
            }
            disconnect();
        }
        
        return found;
    }


    public boolean unblockUser( UserInfo ui ) {
        boolean found = false;

        if ( connect() ) {
            String sql = "UPDATE " + TABLE_PREFIX + "users SET blocked='f' WHERE upper(blocked)='T' AND user_name!='root' AND user_name='" + 
                ui.getUserName() + "'";
            try {
                found = con.prepareStatement(sql).executeUpdate() > 0;
            }
            catch ( Exception e ) {
                found = false;
            }
            disconnect();
        }
        
        return found;
    }


    public String toSQL( String tree, String subtree ) {
        String sql = null;
        
        if ( tree.trim().equalsIgnoreCase("users") ) {
            sql = "SELECT * FROM " + TABLE_PREFIX + "users";
            if ( subtree != null ) sql += " WHERE user_name='" + subtree + "'";
        }
        
        if ( tree.trim().equalsIgnoreCase("groups") ) {
            if ( subtree == null ) 
                sql = "all;SELECT group_name FROM " + TABLE_PREFIX + "groups";
            else {
                if ( subtree.toLowerCase().startsWith("user.") ) {
                    String uname = subtree.substring( subtree.indexOf(".") + 1 );
                    sql = "group;SELECT group_name FROM " + TABLE_PREFIX + "users, " + TABLE_PREFIX + "ug_relation, " + TABLE_PREFIX + 
                        "groups" + " WHERE user_name='" + uname + "'" + 
                        " AND " + TABLE_PREFIX + "users.user_id=" + TABLE_PREFIX + "ug_relation.user_id" + 
                        " AND " + TABLE_PREFIX + "ug_relation.group_id=" + TABLE_PREFIX + "groups.group_id"; 
                }
        
                if ( subtree.toLowerCase().startsWith("group.") ) {
                    String gname = subtree.substring( subtree.indexOf(".") + 1 );
                    sql = "user;SELECT user_name FROM " + TABLE_PREFIX + "users, " + TABLE_PREFIX + "ug_relation, " + TABLE_PREFIX + 
                        "groups" + " WHERE group_name='" + gname + "'" + 
                        " AND " + TABLE_PREFIX + "users.user_id=" + TABLE_PREFIX + "ug_relation.user_id" + 
                        " AND " + TABLE_PREFIX + "ug_relation.group_id=" + TABLE_PREFIX + "groups.group_id"; 
                }
            }
        }
        if ( sql == null ) 
            errorMsg = tree + " " + LogManager.getReference().getMessage("admin_tree_missing"); 
        
        return sql;
    }

    
    public InputStream execute( String tree, String sql ) {
        StringInputStream sis = new StringInputStream();
        String name = "", status = "";
        int id1 = -1, id2 = -1;
        
        if ( connect() ) {
            try {
        
                if ( tree.trim().equalsIgnoreCase("_admin.groups") ) {
                    status = "all";
                    if ( sql.indexOf(";") != -1 ) {
                        status = sql.substring( 0, sql.indexOf(";") ).toLowerCase();    
                        sql = sql.substring( sql.indexOf(";") + 1 );
                    }
                    
                    Statement stmt = con.createStatement();
                    ResultSet   rs = stmt.executeQuery( sql );
                    StringBuffer sb = null;
                    
                    if ( rs.next() ) {
                        sb = new StringBuffer();
                        sb.append( rs.getString(1) );
                    }
                    while ( rs.next() ) sb.append( ", " + rs.getString(1) );
                    rs.close();stmt.close();
        
                    if ( status.equals("all") )
                        if ( sb != null )
                            sis.write( si.getResponse("info", 0, 
                                                      LogManager.getReference().getMessage("groups") + sb.toString() ) );
                        else
                            sis.write( si.getResponse("error", 0,
                                                      LogManager.getReference().getMessage("no_groups") ) );
                    if ( status.equals("group") )
                        if ( sb != null )
                            sis.write( si.getResponse("info", 0,
                                                      LogManager.getReference().getMessage("user_groups") + sb.toString() ) );
                        else     
                            sis.write( si.getResponse("error", 0,
                                                      LogManager.getReference().getMessage("no_user_groups") ) );
                    if ( status.equals("user") )
                        if ( sb != null )    
                            sis.write( si.getResponse("info", 0,
                                                      LogManager.getReference().getMessage("group_users") + sb.toString() ) );
                        else     
                            sis.write( si.getResponse("error", 0,
								  LogManager.getReference().getMessage("no_group_users") ) );
                }
        
                if ( tree.trim().equalsIgnoreCase("_admin.users") ) {
                    Statement stmt = con.createStatement();
                    ResultSet   rs = stmt.executeQuery( sql );
                    StringBuffer sb = new StringBuffer();
                    int nrReg = 0;

                    while ( rs.next() ) {
                        id1 = rs.getInt("user_id");
                        id2 = rs.getInt("home_dir_id");
                        if ( rs.getString("blocked").equalsIgnoreCase("T") ) status = "blocked";
                        name = rs.getString("user_name");
                        sb.append( name + ", " );
                        nrReg ++;
                    } 
                    rs.close();stmt.close();

                    if ( nrReg == 1 ) {
                        String homeDir = "";
                        String groups  = "";
                        
                        while ( id2 > 0 ) {
                            stmt = con.createStatement();
                            rs   = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "trees WHERE tree_id=" + id2 );
                            if ( rs.next() ) {
                                homeDir = rs.getString("tree_name") + "." + homeDir;
                                id2 = rs.getInt("tree_parent");
                            }
                            rs.close(); stmt.close();
                            if ( homeDir.trim().length() < 1) break;
                        }
                        
                        stmt = con.createStatement();
                        rs = stmt.executeQuery("SELECT group_name FROM " + TABLE_PREFIX + "ug_relation, " + 
                                               TABLE_PREFIX + "groups WHERE " + TABLE_PREFIX + "ug_relation.group_id=" + 
                                               TABLE_PREFIX + "groups.group_id AND user_id=" + id1);
                        while ( rs.next() ) groups += rs.getString("group_name") + ", ";
                        rs.close();stmt.close();
                        
                        sb = new StringBuffer();    
                        sb.append( "     Username : " + name    + "\n");
                        if ( homeDir.lastIndexOf(".") != -1 ) homeDir = homeDir.substring( 0, homeDir.lastIndexOf(".") );
                        sb.append( " Home subtree : " + homeDir + "\n");
                        sb.append( "       Groups : " + groups  + "\n");
                        sb.append( "       Status : " + status  + "\n");
                    }
                
                    if ( nrReg == 0 ) {
                        sis.flush();
                        sis.write( si.getResponse("error", 0, 
                                                  LogManager.getReference().getMessage("user_not_found") ) );
                        return sis.getInputStream();
                    }
                    
                    sis.write( si.getResponse("info", 0, sb.toString() ) );
                }
            }
            catch ( Exception e) {
                errorMsg = LogManager.getReference().getMessage("user_group_list_error");
                LogManager.getReference().message("error", "query_failed", "\nSQL = " + sql + "\n" + e.getMessage()); 
            }   
            disconnect();
        }
        
        return sis.getInputStream();
    }


    private StringInputStream setNamingFlag ( StringInputStream sis, String subtree, int tID, 
                                              String namingType, boolean verbose, boolean recursive, boolean first ) {
        String errorMSG = "";
        String naming   = handler.getNaming( tID );
        
        if ( first ) sis.write( "<response reqno='" + si.getRequestNr() + "'>\n" );
        
        try {
            if ( naming != null && naming.trim().equalsIgnoreCase( namingType.trim() ) ) {
                errorMSG = "'" + subtree + "' " + LogManager.getReference().getMessage("naming_flag_exist") +
                    " '" + namingType + "'.";
                sis.write("<additional>\n" +
                          "<info code=\"0\">\n" + 
                          errorMSG + "\n" +
                          "</info>\n" +
                          "</additional>\n");
            }
            else {                    
                PreparedStatement ps = null;

                ps = con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET naming= ? , modify_time= ? WHERE tree_id=" + tID);
                if ( namingType.trim().equalsIgnoreCase("Manual") )
                    ps.setString   ( 1, "M");
                else 
                    ps.setString   ( 1, "A");
                ps.setTimestamp( 2, new Timestamp( System.currentTimeMillis() ) );
                ps.executeUpdate();ps.close();

                errorMSG = "'" + subtree + "' " + LogManager.getReference().getMessage("subtree_set_flag") +
                    " '" + namingType + "'.";
		
                sis.write("<additional>\n" +
                          "<info code=\"0\">\n" + 
                          errorMSG + "\n" +
                          "</info>\n" +
                          "</additional>\n");
            }
            
            if ( recursive ) {
                Statement stmt = con.createStatement();
                ResultSet rs   = stmt.executeQuery("SELECT tree_id, tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID );
                while ( rs.next() ) {
                    String stree = subtree.trim();
                    if ( !stree.endsWith("/") ) stree += "/";
                    stree += rs.getString("tree_name");
                    
                    if ( !handler.hasTreeRights("read", stree, false ) )
                        sis.write("<error code=\"0\">\n" +
                                  LogManager.getReference().getMessage("src_read_denied") + " '" + stree + "'.\n" +
                                  "</error>\n");
                    else if ( !handler.hasTreeRights("write", stree, false ) ) 
                        sis.write("<error code=\"0\">\n" + 
                                  LogManager.getReference().getMessage("dest_write_denied") + " '" + stree + "'.\n" +
                                  "</error>\n");
                    else sis = setNamingFlag( sis, stree, rs.getInt("tree_id"), namingType, verbose, recursive, false );
                }
                rs.close();stmt.close();
            }
        }       
        catch ( Exception ex ) {
            errorMSG = "'" + subtree + "' subtree naming flag " + 
                LogManager.getReference().getMessage("subtree_not_converted") +" '" + namingType + "'.";
            sis.write("<error code=\"0\">\n" + errorMSG + "\n" + "</error>\n");
        }

        if ( first ) sis.write("</response>\n");

        return sis;
    }


    private StringInputStream setStoreType ( StringInputStream sis, String subtree, int tID, 
                                             String storeType, boolean verbose, boolean recursive, boolean first ) {
        String  dtdName = null;
        String      sql = "SELECT doc_id, doc_name FROM " + TABLE_PREFIX + "repository WHERE order_nr=1 AND upper(status)<>'D' AND tree_id=" + 
            tID + " ORDER BY 1 ASC";
        String errorMSG = "";
        int       docNr = 0;
        String     mode = handler.getMode( tID );
	
        if ( first ) sis.write( "<response reqno='" + si.getRequestNr() + "'>\n" );
	
        try {
            ResultSet   rs = null;
            Statement stmt = null;
            if ( mode != null && mode.trim().equalsIgnoreCase( storeType.trim() ) ) {
                errorMSG = "'" + subtree + "' " + LogManager.getReference().getMessage("store_mode_exist") +
                    " '" + storeType + "' mode.";
	
                sis.write("<additional>\n" +
                          "<info code=\"0\">\n" + 
                          errorMSG + "\n" +
                          "</info>\n" +
                          "</additional>\n");
            }
            else {
                stmt = con.createStatement();
                rs   = stmt.executeQuery( sql );
                while ( rs.next() ) {
                    int dID = rs.getInt("doc_id");
                    String docName = rs.getString("doc_name");;
                    InputStream xml = handler.getDocument( tID, dID );
                    
                    if ( !storeType.toLowerCase().startsWith("free") ) {
                        dtdName = si.getDTDName( xml );
                        xml     = new SepInputStream( xml, si.getHeader(), "" );
                        
                        if ( dtdName == null ) 
                            errorMSG += "'" + subtree + "/" + dID + "' " + 
                                LogManager.getReference().getMessage("conversion_failed") +
                                " '" + storeType + "' mode." +  
                                LogManager.getReference().getMessage("not_xml") + "\n";
                    }
		
                    if ( dtdName != null && storeType.equalsIgnoreCase("flexible") ) {
                        if ( !si.createDTDFromXML( subtree, docName, dtdName, xml, tID, dID, rdbInfo.getDTDDirInfo(), false ) )
                            errorMSG += "'" + subtree + "/" + dID + "' " + 
                                LogManager.getReference().getMessage("conversion_failed") +
                                " '" + storeType + "' mode." + si.getErrorMSG() + "\n";
                    }
                    
                    if ( dtdName != null && storeType.equalsIgnoreCase("strict") ) {
                        if ( !si.validateXML( subtree, docName, dtdName, xml, tID, dID , false ) )
                            errorMSG += "'" + subtree + "." + dID + "' " + 
                                LogManager.getReference().getMessage("conversion_failed") +
                                " '" + storeType + "' mode." + si.getErrorMSG() + "\n";
                    }
                    docNr++;
                }
                rs.close();stmt.close();
                
                if ( errorMSG.length() > 0 ) {
                    sis.write("<error code=\"0\">\n" + 
                              errorMSG + "\n" +
                              "</error>\n");
                }
                else {
                    PreparedStatement ps = null;

                    ps = con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET store_mode= ? , modify_time= ? WHERE tree_id=" + tID);
                    ps.setString   ( 1, storeType);
                    ps.setTimestamp( 2, new Timestamp( System.currentTimeMillis() ) );
                    ps.executeUpdate();ps.close();
		
                    ps = con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET status= ? , modify_time= ? WHERE upper(status)<>'D' AND tree_id=" + tID );
                    ps.setString   ( 1, "U");
                    ps.setTimestamp( 2, new Timestamp( System.currentTimeMillis() ) );
                    ps.executeUpdate();ps.close();
		
                    if ( verbose ) 
                        errorMSG = docNr + " " + LogManager.getReference().getMessage("docs_converted") +
                            " '" + storeType + "' mode into the '" + subtree + "'.";
                    else
                        errorMSG = "'" + subtree + "' " + LogManager.getReference().getMessage("subtree_converted") +
                            " '" + storeType + "' mode.";
		
                    sis.write("<additional>\n" +
                              "<info code=\"0\">\n" + 
                              errorMSG + "\n" +
                              "</info>\n" +
                              "</additional>\n");
                }
            }
            
            if ( recursive ) {
                stmt = con.createStatement();
                rs   = stmt.executeQuery("SELECT tree_id, tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID );
                while ( rs.next() ) {
                    String stree = subtree.trim();
                    if ( !stree.endsWith(".") ) stree += ".";
                    stree += rs.getString("tree_name");
                    
                    if ( !handler.hasTreeRights("read", stree, false ) )
                        sis.write("<error code=\"0\">\n" +
                                  LogManager.getReference().getMessage("src_read_denied") + " '" + stree + "'.\n" +
                                  "</error>\n");
                    else if ( !handler.hasTreeRights("write", stree, false ) ) 
                        sis.write("<error code=\"0\">\n" + 
                                  LogManager.getReference().getMessage("dest_write_denied") + " '" + stree + "'.\n" +
                                  "</error>\n");
                    else sis = setStoreType( sis, stree, rs.getInt("tree_id"), storeType, verbose, recursive, false );
                }
                rs.close();stmt.close();
            }
        }
        catch ( Exception ex ) {
            errorMSG = "'" + subtree + "' " + LogManager.getReference().getMessage("subtree_not_converted") +
                " '" + storeType + "' mode.";
            
            sis.write("<error code=\"0\">\n" + 
                      errorMSG + "\n" +
                      "</error>\n");
        }
        
        if ( first ) sis.write("</response>\n");
        
        return sis;
    }


    private StringInputStream copyTree( StringInputStream sis, String dest, String src, 
                                        int srcID, boolean recursive, boolean verbose , boolean first ) {
        String   sql = "SELECT doc_id FROM " + TABLE_PREFIX + "repository WHERE (status='S' OR status='W' OR status='U') AND order_nr=1 AND tree_id=" + srcID;
        ResultSet rs = null;
        Statement stmt = null;
        String error = null;
        int    docNr = 0;

        try {
            if ( first ) 
                sis.write("<response reqno='" + si.getRequestNr() + "'>\n" +
                          "<additional>\n" +
                          "<info code=\"0\">\n");
            stmt = con.createStatement();
            rs = stmt.executeQuery( sql );
            while ( rs.next() && error == null ) {
                int       dID = rs.getInt("doc_id");
                String srcDoc = src + "/" + dID;
                
                if ( handler.hasDocumentRights("read", srcDoc ) ) {
                    if ( ( error = copyDocument( dest, srcID, dID ) ) == null ) {
                        if ( verbose ) sis.write( srcDoc + " -&gt; " + dest + "\n"); docNr ++;
                    }
                    else sis.write ( LogManager.getReference().getMessage("doc_copy_failed") + " " + 
                                     srcDoc + ": " + error + "\n");
                }
                else sis.write ( LogManager.getReference().getMessage("doc_copy_failed") + " " + srcDoc + ": " + 
                                 LogManager.getReference().getMessage("permission_denied") + "!" + "\n"); 
            }
            rs.close();stmt.close();
            
            if ( !verbose || docNr == 0 ) 
                sis.write ( docNr + " " + LogManager.getReference().getMessage("file_copied") + " '" + dest + "'.\n");
            
            if ( recursive ) {
                sql = "SELECT tree_id, tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + srcID;
                stmt = con.createStatement();
                rs   = stmt.executeQuery( sql );
                
                while ( rs.next () ) {
                    int      tID = rs.getInt("tree_id");
                    String tName = rs.getString("tree_name");
                    
                    if ( handler.hasTreeRights("read", src + "/" + tName , false) ) {
                        sis = copyTree ( sis, dest + "/" + tName, src + "/" + tName, tID, recursive, verbose, false );
                    }
                    else
                        sis.write ( LogManager.getReference().getMessage("src_read_denied") + " '" + src + "/" + tName + "'.\n");
                    
                }
                rs.close();stmt.close();
            } 
            
            if ( first )
                sis.write( "</info>\n" +
                           "</additional>\n" +
                           "</response>\n" );

        }
        catch ( Exception e ) {
            sis.flush();
            sis.write ( si.getResponse("error", 0, src + " " + 
				    LogManager.getReference().getMessage("subtree_copy_error") + " " + dest + "\n") );
        }
        
        return sis;
    }

    
    private String copyDocument( String dest, int srcTreeID, int srcDocID ) {
        return si.writeDocument( dest, handler.getDocumentName( srcTreeID, srcDocID ), 
                                 handler.getDocument( srcTreeID, srcDocID ), 
                                 handler.getMode( handler.getTreeID( dest ) ), 
                                 -1, false, "write", rdbInfo , false);
    }


    private Vector longFormatTrees( String filter ) {
        String sql 
            = " SELECT tree_id, tree_name, user_read, user_write, group_read, group_write, other_read, other_write, user_name, group_name, modify_time, store_mode, naming" +
            " FROM " + TABLE_PREFIX + "trees, " + TABLE_PREFIX + "users, " + TABLE_PREFIX + "groups WHERE " + filter + " AND " + 
            TABLE_PREFIX + "trees.user_id=" + TABLE_PREFIX + "users.user_id AND " + 
            TABLE_PREFIX + "trees.group_id=" + TABLE_PREFIX + "groups.group_id ORDER BY 2 ASC";
        Vector result = null;

        try {   
            Statement stmt = con.createStatement();
            ResultSet   rs = stmt.executeQuery( sql );
            
            while ( rs.next() ) {
                LongFormat  lf = new LongFormat();
                String  rights = "t";
        
                if ( result == null ) result = new Vector();
                if ( rs.getString("user_read").equalsIgnoreCase("T") )   rights += "r"; else rights += "-";
                if ( rs.getString("user_write").equalsIgnoreCase("T") )  rights += "w"; else rights += "-";
                if ( rs.getString("group_read").equalsIgnoreCase("T") )  rights += "r"; else rights += "-";
                if ( rs.getString("group_write").equalsIgnoreCase("T") ) rights += "w"; else rights += "-";
                if ( rs.getString("other_read").equalsIgnoreCase("T") )  rights += "r"; else rights += "-";
                if ( rs.getString("other_write").equalsIgnoreCase("T") ) rights += "w"; else rights += "-";
        
                lf.setRights( rights );
                lf.setUserName( rs.getString("user_name") );
                lf.setGroupName( rs.getString("group_name") );
                
                String mode = rs.getString("store_mode");
                for ( int i = mode.length(); i < 11; i++ ) mode += " "; 
                
                String naming = "Auto";
                if ( rs.getString("naming").equals("M") ) naming = "Manual";
                for ( int i = naming.length(); i < 7; i++ ) naming += " "; 
                
                String timeStamp = null;
                try {
                    timeStamp = rs.getTimestamp("modify_time").toString();
                }
                catch ( Exception e ){
                    timeStamp = rs.getString("modify_time");
                }
                lf.setTimeStamp( mode + naming + timeStamp );
                lf.setHomeDir( "/" + getHome( rs.getInt("tree_id") ) + "/" );
                result.addElement(lf);
            } 
            rs.close();stmt.close();
        } 
        catch ( Exception e ) {
            LogManager.getReference().message("error", "query_failed", ":" + sql + ".\n" + e.getMessage() );
            return null;
        }

        return result;
    }

    
    private Vector longFormatDocs( String filter ) {
        String sql = " SELECT tree_id, doc_id, doc_name, user_read, user_write, group_read, group_write, other_read, other_write, user_name, group_name, modify_time " +
            " FROM " + TABLE_PREFIX + "repository, " + TABLE_PREFIX + "users, " + TABLE_PREFIX + "groups WHERE order_nr=1 AND upper(status)<>'D' AND " + filter + 
            " AND " + TABLE_PREFIX + "repository.user_id=" + TABLE_PREFIX + "users.user_id AND " + TABLE_PREFIX + "repository.group_id=" + 
            TABLE_PREFIX + "groups.group_id ORDER BY 2 ASC";
        Vector result = null;

        try {
            Statement stmt = con.createStatement();   
            ResultSet   rs = stmt.executeQuery( sql );

            while ( rs.next() ) {
                LongFormat lf = new LongFormat();
                String rights = "-";
        
                if ( result == null ) result = new Vector();
                if ( rs.getString("user_read").equalsIgnoreCase("T") )   rights += "r"; else rights += "-";
                if ( rs.getString("user_write").equalsIgnoreCase("T") )  rights += "w"; else rights += "-";
                if ( rs.getString("group_read").equalsIgnoreCase("T") )  rights += "r"; else rights += "-";
                if ( rs.getString("group_write").equalsIgnoreCase("T") ) rights += "w"; else rights += "-";
                if ( rs.getString("other_read").equalsIgnoreCase("T") )  rights += "r"; else rights += "-";
                if ( rs.getString("other_write").equalsIgnoreCase("T") ) rights += "w"; else rights += "-";
        
                lf.setRights( rights );
                lf.setUserName( rs.getString("user_name") );
                lf.setGroupName( rs.getString("group_name") );
                String timeStamp = null;
                try {
                    timeStamp = rs.getTimestamp("modify_time").toString();
                }
                catch ( Exception e ){
                    timeStamp = rs.getString("modify_time");
                }
                lf.setTimeStamp( timeStamp );
                lf.setHomeDir( "/" + getHome( rs.getInt("tree_id") ) + "/" + rs.getString("doc_name") );
                result.addElement(lf);
            }
            rs.close();stmt.close();
        } 
        catch ( Exception e ) {
            LogManager.getReference().message("error", "query_failed", ":" + sql + ".\n" + e.getMessage() );
            return null;
        }

        return result;
    }


    private String getHome( int tID ) throws Exception {
        String  path = "";
        Statement stmt = con.createStatement();
        ResultSet  rsx = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "trees WHERE tree_id=" + tID );
        
        if ( rsx.next () ) {
            int     pID = rsx.getInt("tree_parent");
            String name = rsx.getString("tree_name");
            if ( pID > 0 ) path = getHome( pID ) + "/";
            path += name;
        }
        rsx.close();stmt.close();
        
        return path;
    }


    private boolean removeTreeRecursively( String tName ) {
        int tID = handler.getTreeID( tName );
         
        if ( tID == -1 || ( !hasRights() && user_id != handler.getOwnerID( tID ) ) ) return true;
        
        try {
            boolean error = false;
            Statement stmt = con.createStatement();
            ResultSet   rs = stmt.executeQuery("SELECT tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID);
            while ( rs.next() && !error )
                error = removeTreeRecursively( tName + "." + rs.getString("tree_name") );
            rs.close();stmt.close();
            if ( !error ) {
                con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET status='D' WHERE tree_id=" + tID).execute();
                (new LexiconManager()).runLexiconManager( false , true);
                return false;
            }
            return true;        
        }
        catch ( Exception e ) {
            return true;
        }
        
    }
    

    private String chmodForDocument(int tID, int dID, String mode, boolean verbose, String doc) {
        int       uid = getDocumentUserID(tID, dID);
        StringTokenizer st = new StringTokenizer(mode, "ugoa+-=rw,", false);
        String result = "";
                
        if ( user_id != -1 && user_id != uid ) { 
            LogManager.getReference().message("warn" , "chmod_doc.fail", " '" + uName + ",.", "' " + doc + "' "); 
            result += "\n" + LogManager.getReference().getMessage("chmod_doc_failed") + " '" + doc + "'";
        }
        
        if ( st.hasMoreTokens() ) { 
            errorMsg = LogManager.getReference().getMessage("mode_error") + " '" + mode + "'";
            return null;
        }
        
        if ( setDocumentMode(tID, dID, mode) ) {
            LogManager.getReference().message("info" , "chmod_doc.ok", " '" + uName + ",.", "' " + doc + "' "); 
            if ( verbose ) result +=  "\n" + "'" + doc + "' " + LogManager.getReference().getMessage("chmod_doc");
        }
        else {
            LogManager.getReference().message("warn" , "chmod_doc.fail", " '" + uName + ",.", "' " + doc + "' "); 
            result += "\n" + LogManager.getReference().getMessage("chmod_doc_failed") + " '" + doc + "'";
        }

        return result; 
    }


    private String chmodForTree(int tID, String mode, boolean recursive, boolean verbose, String doc) {
        String result = "";
        StringTokenizer st = new StringTokenizer(mode, "ugoa+-=rw,", false);

        if ( st.hasMoreTokens() ) {
            errorMsg = LogManager.getReference().getMessage("mode_error") + " '" + mode + "'";
            return null;
        }
        result = chmodRecursively(tID, mode, recursive, verbose, doc);
        if ( result == null ) return null;
        if ( result.trim().length() < 1 ) result = LogManager.getReference().getMessage("chmod_successfull");

        return result;
    }

    
    private String chmodRecursively(int tID, String mode, boolean recursive, boolean verbose, String doc) {
        int      _uID = getTreeUserID( tID );
        String result = "";
        boolean accessDenied = false;
        
        if ( user_id != -1 && user_id != _uID ) accessDenied = true; 

        if ( recursive ) {
            try {
                Statement stmt = con.createStatement();
                ResultSet   rs = stmt.executeQuery("SELECT tree_id, tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID);
                while ( rs.next() ) {
                    int      _tid = rs.getInt("tree_id");
                    String _tname = rs.getString("tree_name"), dot = "";
                    
                    if ( !doc.endsWith("/") ) dot += "/";
                    result += chmodRecursively(_tid, mode, recursive, verbose, doc + dot + _tname);
                }
                rs.close();stmt.close();
            
                if ( !accessDenied ) {
                    stmt = con.createStatement();
                    rs   = stmt.executeQuery("SELECT doc_id FROM " + TABLE_PREFIX + "repository WHERE order_nr=1 AND tree_id=" + tID );
            	    while ( rs.next () ) {
                	int _did = rs.getInt("doc_id");
                	String dot = "";
                    
                	if ( !doc.endsWith("/") ) dot += "/";
                	result += chmodForDocument(tID, _did, mode, verbose, doc + dot + _did );
            	    }
            	    rs.close();stmt.close();
                }
            } 
            catch ( Exception e ) {}
        }
        
        if ( !accessDenied && setTreeMode( tID, mode, recursive ) ) {
            if ( verbose ) result += "\n" + "'" + doc + "' " + LogManager.getReference().getMessage("chmod_tree");
            LogManager.getReference().message("info" , "chmod_tree.ok", " '" + uName + ",.", "' " + doc + "' "); 
        }
        else {
            result += "\n" + LogManager.getReference().getMessage("chmod_tree_failed") + " '" + doc + "'";
            LogManager.getReference().message("warn" , "chmod_tree.fail", " '" + uName + ",.", "' " + doc + "' "); 
        }

        return result;
    }


    private boolean setTreeMode( int tID, String mode, boolean verify ) {
        boolean    changed = true;
        StringTokenizer st = new StringTokenizer(mode, ",");
        setAutoCommit( false );
        try {
            while ( st.hasMoreTokens() ) {
                String sql = modeToSQL( st.nextToken() );
                if ( sql == null ) { 
                    changed = false; break; 
                }
                if( verify && user_id != -1 && sql.indexOf("user_read='f'") != -1 ) { 
                    changed = false; break; 
                }
                con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET " + sql + " WHERE tree_id=" + tID).execute();
            }
            if ( changed ) con.commit(); else con.rollback();
        }       
        catch ( Exception e ) {
            changed = false;
        }
        setAutoCommit( true );
                        
        return changed;
    }


    private boolean setDocumentMode( int tID, int dID, String mode ) {
        boolean    changed = true;
        StringTokenizer st = new StringTokenizer(mode, ",");
        
        setAutoCommit( false );
        try {
            while ( st.hasMoreTokens() ) {
                String sql = modeToSQL( st.nextToken() );
                if ( sql == null ) { changed = false; break; }
                sql = "UPDATE " + TABLE_PREFIX + "repository SET " + sql + " WHERE tree_id=" + tID + " AND doc_id=" + dID;
                con.prepareStatement(sql).execute();
            }
            if ( changed ) con.commit(); else con.rollback();
        }       
        catch ( Exception e ) {
            changed = false;
        }
        setAutoCommit( true );
                        
        return changed;
    }
    

    private void setAutoCommit( boolean flag ) {
        try {
            con.setAutoCommit(flag);
        }
        catch ( Exception e ) {}
    }

    
    private String modeToSQL( String mode ) {
        boolean    first = true, firstOp = true, error = false;
        boolean    hasUser = false, hasGroup = false, hasOther = false;
        char       modifier = '\0';
        StringTokenizer st = new StringTokenizer(mode, "ugoa+-=rw", true);
        String     sql = "";
                
        while ( st.hasMoreTokens() && !error ) {
            char op = ( st.nextToken() ).charAt(0);
            
            if ( first && ( op == 'r' || op == 'w' ) ) {
                error = true;
                continue;
            } 
            
            if ( first && ( op == '+' || op == '-' || op == '=' ) ) {
                hasUser  = true; 
                hasGroup = true; 
                hasOther = true;
                modifier = op;
                first = false;
                continue;
            }
            
            first = false;
            
            switch ( op ) {
            case 'u' : hasUser  = true; break;
            case 'g' : hasGroup = true; break;
            case 'o' : hasOther = true; break;
            case 'a' : hasUser  = true; 
                hasGroup = true; 
                hasOther = true; break;
            case 'r' : 
                switch ( modifier ) {
                case '+' : sql = addOption( sql, hasUser, hasGroup, hasOther , firstOp, "read", "t");
                    firstOp = false;
                    break;
                case '-' : sql = addOption( sql, hasUser, hasGroup, hasOther , firstOp, "read", "f");
                    firstOp = false;
                    break;
                case '=' : sql = addEqualOption ( sql, hasUser, hasGroup, hasOther , firstOp, "t", "f");
                    firstOp = false;
                    break;
                default  : error = true;        
                } 
                break;

            case 'w' : 
                switch ( modifier ) {
                case '+' : sql = addOption( sql, hasUser, hasGroup, hasOther , firstOp, "write", "t");
                    firstOp = false;
                    break;
                case '-' : sql = addOption( sql, hasUser, hasGroup, hasOther , firstOp, "write", "f");
                    firstOp = false;
                    break;
                case '=' : sql = addEqualOption ( sql, hasUser, hasGroup, hasOther , firstOp, "f", "t");  
                    firstOp = false;
                    break;    
                default  : error = true;        
                }
                break;

            default  : modifier = op;
            }              
        }
        if ( error ) return null;

        return sql;
    }

    
    private String addOption( String sql, boolean isUser, boolean isGroup, boolean isOther, 
                              boolean first, String right, String flag ) {
        if ( isUser ) {
            sql = addOption( sql, " user_"  + right + "='" + flag + "'", first );first = false;
        }
        if ( isGroup ) {
            sql = addOption( sql, " group_" + right + "='" + flag + "'", first );first = false;
        }
        if ( isOther ) {
            sql = addOption( sql, " other_" + right + "='" + flag + "'", first );first = false;
        }
        return sql;
    } 
    

    private String addEqualOption ( String sql, boolean isUser, boolean isGroup, boolean isOther, 
                                    boolean first, String readFlag, String writeFlag ) {
        if ( isUser )  {
            sql = addOption( sql, " user_read='"   + readFlag  + "'", first );first = false;
            sql = addOption( sql, " user_write='"  + writeFlag + "'", first );
        }
        if ( isGroup ) {
            sql = addOption( sql, " group_read='"  + readFlag  + "'", first );first = false;
            sql = addOption( sql, " group_write='" + writeFlag + "'", first );
        }
        if ( isOther ) {
            sql = addOption( sql, " other_read='"  + readFlag  + "'" , first );first = false;
            sql = addOption( sql, " other_write='" + writeFlag + "'", first );
        } 
        return sql;     
    }
    
    private String addOption ( String src, String str, boolean first ) {
        if ( !first ) src += ",";
        src += str;
        return src;
    } 


    private String chownForDocument(int tID, int dID, int uID, int gID, boolean verbose, String doc) {
        int      _uID = getDocumentUserID(tID, dID);
        String result = "";
        
        if ( user_id != -1 && user_id != _uID ) {
            LogManager.getReference().message("warn" , "chown_doc.fail", " '" + uName + ",.", "' " + doc + "' "); 
            errorMsg = LogManager.getReference().getMessage("chown_doc_failed") + " '" + doc + "'\n";
            return null;
        }
        
        if ( setDocumentOwner(tID, dID, uID, gID) ) {
            if ( verbose ) result +=  "'" + doc + "' " + LogManager.getReference().getMessage("chown_doc") + "\n";
            LogManager.getReference().message("warn" , "chown_doc.ok", " '" + uName + ",.", "' " + doc + "' ");
        }
        else {
            LogManager.getReference().message("warn" , "chown_doc.fail", " '" + uName + ",.", "' " + doc + "' ");
            errorMsg = LogManager.getReference().getMessage("chown_doc_failed") + " '" + doc + "'\n";
            return null;
        }

        return result; 
    } 

    
    private String chownForTree(int tID, int uID, int gID, boolean recursive, boolean verbose, String doc) {
        String result = chownRecursively(tID, uID, gID, recursive, verbose, doc);
        
        if ( result == null ) return null;
        if ( result.trim().length() < 1 ) result = LogManager.getReference().getMessage("chown_successfull");
        LogManager.getReference().message("warn" , "chown_doc.ok", " '" + uName + ",.", "' " + doc + "' ");
        
        return result;
    }

    
    private String chownRecursively(int tID, int uID, int gID, boolean recursive, boolean verbose, String doc) {
        int      _uID = getTreeUserID( tID );
        String result = "";
        boolean accessDenied = false;
        
        if ( user_id != -1 && user_id != _uID ) accessDenied = true;

        if ( recursive ) {
            try {
                Statement stmt = con.createStatement();
                ResultSet   rs = stmt.executeQuery("SELECT tree_id, tree_name FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID);
                while ( rs.next() ) {
                    int      _tid = rs.getInt("tree_id");
                    String _tname = rs.getString("tree_name"), dot = "";
                    
                    if ( !doc.endsWith("/") ) dot += "/";
                    result += chownRecursively(_tid, uID, gID, recursive, verbose, doc + dot + _tname);
                }
                rs.close();stmt.close();
        	
                if ( !accessDenied ) {
                    stmt = con.createStatement();
            	    rs   = stmt.executeQuery("SELECT doc_id FROM " + TABLE_PREFIX + "repository WHERE order_nr=1 AND tree_id=" + tID );
            	    while ( rs.next () ) {
                	int _did = rs.getInt("doc_id");
                	String dot = "";
                    
                	if ( !doc.endsWith("/") ) dot += "/";
                	result += chownForDocument(tID, _did, uID, gID, verbose, doc + dot + _did );
            	    }
            	    rs.close();stmt.close();
                }
            }
            catch ( Exception e ) {
            }
        }
        
        if ( !accessDenied && setTreeOwner(tID, uID, gID) ) {
            if ( verbose ) result += "'" + doc + "' " + LogManager.getReference().getMessage("chown_tree") + "\n";
            LogManager.getReference().message("warn" , "chown_doc.ok", " '" + uName + ",.", "' " + doc + "' ");
        }
        else {
            result += LogManager.getReference().getMessage("chown_tree_failed") + " '" + doc + "'\n";
            LogManager.getReference().message("warn" , "chown_doc.fail", " '" + uName + ",.", "' " + doc + "' "); 
        }
        
        return result;
    }
    

    private boolean setTreeOwner( int tID, int uID, int gID ) {
        boolean changed = false;
        
        String sql = "UPDATE " + TABLE_PREFIX + "trees SET";
                 
        if ( uID != -2 ) sql += " user_id=" + uID;
        if ( gID != -2 ) {
            if ( uID != -2 ) sql += " ,";
            sql += " group_id=" + gID; 
        }
        sql += " WHERE tree_id=" + tID;
        try {
            con.prepareStatement(sql).execute();
            changed = true;
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "query_failed", "\nSQL = " + sql + "\n" + e.getMessage() );
        }
        
        return changed;
    }


    private boolean setDocumentOwner( int tID, int dID, int uID, int gID ) {
        boolean changed = false;
        
        String sql = "UPDATE " + TABLE_PREFIX + "repository SET";
                 
        if ( uID != -2 ) sql += " user_id=" + uID;
        if ( gID != -2 ) {
            if ( uID != -2 ) sql += " ,";
            sql += " group_id=" + gID; 
        }
        sql += " WHERE tree_id=" + tID + " AND doc_id=" + dID;
        try {
            con.prepareStatement(sql).execute();
            changed = true;
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "query_failed", "\nSQL = " + sql + "\n" + e.getMessage() );
        }
        
        return changed;
    }


    private int getTreeUserID(int tID) {
        int uID = -1;
        
        try {
            Statement stmt = con.createStatement();
            ResultSet   rs = stmt.executeQuery("SELECT user_id FROM " + TABLE_PREFIX + "trees WHERE tree_id=" + tID);
            if ( rs.next() ) uID = rs.getInt("user_id");
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "subtree_user_error", e.getMessage() ); 
        } 
        
        return uID;
    }
    

    private int getDocumentUserID(int tID, int dID) {
        int uID = -1;
        
        try {
            Statement stmt = con.createStatement();
            ResultSet   rs = stmt.executeQuery("SELECT user_id FROM " + TABLE_PREFIX + "repository WHERE order_nr=1 AND tree_id=" + 
                                               tID + " AND doc_id=" + dID);
            if ( rs.next() ) uID = rs.getInt("user_id");
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "document_user_error", e.getMessage() ); 
        } 
        
        return uID;
    }


    private void updateGroup( int gID, String name ) throws Exception {
        try {
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "groups SET group_name='" + name + "' WHERE group_id=" + gID).execute();
        }
        catch ( Exception e ) {
            throw e;
        }
    }

     
    private void deleteGroup( int gID ) throws Exception {
        try {
            con.prepareStatement("DELETE FROM " + TABLE_PREFIX + "groups WHERE group_id=" + gID).execute();
        } 
        catch ( Exception e ) {
            throw e;
        } 
    }
    

    private boolean groupHasUsers( int gID ) throws Exception {
        boolean found = false;
        
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery ("SELECT * FROM " + TABLE_PREFIX + "ug_relation WHERE group_id=" + gID);
            found = rs.next();
            rs.close();stmt.close(); 
        } 
        catch (Exception e) {
            throw e;
        }
        
        return found;
    }
    

    private void removeUser( int uID ) throws Exception {
        try {
    	    deleteHomeDir ( uID );
            deleteUser( uID );
            deleteUserGroups ( uID );    
    	    assignUserDocsToRoot( uID );
        } 
        catch ( Exception ex ) {
            throw ex;
        }
    }
    
    
    private void assignUserDocsToRoot( int uID ) throws Exception {
        try {
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET user_id=0, group_id=0 WHERE user_id=" + uID).execute();
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET user_id=0, group_id=0 WHERE user_id=" + uID).execute();
        } 
        catch ( Exception e ) {
            throw e;
        }
    }
    
    
    private void removeHomeDirRecursively ( int tID ) throws Exception {
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT tree_id FROM " + TABLE_PREFIX + "trees WHERE tree_parent=" + tID );
            while ( rs.next() ) removeHomeDirRecursively( rs.getInt("tree_id") ); rs.close(); stmt.close();
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET status='D' WHERE tree_id=" + tID).execute();
        }
        catch ( Exception e ) {
            throw e;
        }
    }
    
    
    private void deleteHomeDir( int uID ) throws Exception {
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT home_dir_id FROM " + TABLE_PREFIX + "users WHERE user_id=" + uID);
            
            if ( rs.next() ) {
                int tID = rs.getInt("home_dir_id");
	
                if ( tID > 0 ) {
                    removeHomeDirRecursively( tID );
                    handler.deleteTree( tID );
                    (new LexiconManager()).runLexiconManager( false , true); 
                }
            } rs.close(); stmt.close();
        } 
        catch ( Exception e ) {
            throw e;
        }
    }
    

    private void deleteUserGroups ( int uID ) throws Exception {
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "ug_relation WHERE user_id=" + uID );
            while ( rs.next() ) {
                int gID = rs.getInt("group_id");
                String sql = "UPDATE " + TABLE_PREFIX + "ug_relation SET group_id=" + gID + " WHERE group_id=" + gID;
                if ( con.prepareStatement(sql).executeUpdate() < 2 )
                    con.prepareStatement("DELETE FROM " + TABLE_PREFIX + "groups WHERE group_id=" + gID).execute();
            }
            rs.close();stmt.close();
            con.prepareStatement("DELETE FROM " + TABLE_PREFIX + "ug_relation WHERE user_id=" + uID ).execute();
        }
        catch ( Exception e ) {
            throw e;
        }
    }

    
    private void deleteUser( int uID ) throws Exception{
        try {
            con.prepareStatement("DELETE FROM " + TABLE_PREFIX + "users WHERE user_id=" + uID).execute();
        } 
        catch ( Exception e ) {
            throw e;
        }
    }

    
    private boolean userExist(String uname) {
        boolean found = false;  
    
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "users WHERE user_name='" + uname + "'");
            found = rs.next(); 
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "user_exist_error", uname); 
        }
        
        return found;
    }


    private int getGroupID(int uID) {
        int id = -1;
        
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT group_id FROM " + TABLE_PREFIX + "ug_relation where user_id=" + uID);
            if ( rs.next() ) id = rs.getInt("group_id");
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "group_user_id_error", "" + uID); 
        }
        
        return id;
    }
    

    private void moveDocuments(int oldID, int newID) {
        try {
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "trees SET  tree_parent=" + newID + " WHERE tree_parent=" + oldID ).execute();
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "structs    SET tree_id=" + newID + " WHERE tree_id=" + oldID ).execute();
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "repository SET tree_id=" + newID + " WHERE tree_id=" + oldID ).execute();
            con.prepareStatement("UPDATE " + TABLE_PREFIX + "barrels    SET tree_id=" + newID + " WHERE tree_id=" + oldID ).execute();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "move_document_error", e.getMessage()); 
        }
    }
    

    private int updateUser( String lname, String uname, String passwd, int homeID ) throws Exception {
        int oldHomeId = -1;
        String    sql = "UPDATE " + TABLE_PREFIX + "users SET ";
        boolean first = true;
        
        if ( uname != null ) {
            sql += " user_name='" + uname + "'";
            first = false;
        }
        if ( passwd != null ) {
            if ( !first ) sql += " ,";
            sql += " user_passwd='" + passwd + "'";
            first = false; 
        }
        
        if ( homeID != -1 ) {
            if ( !first ) sql += " ,";
            sql += " home_dir_id=" + homeID;
            oldHomeId =  getOldHomeID( lname );
            first = false; 
        }
        
        sql += " WHERE user_name='" + lname + "'";
        if ( first ) return -1;

        try {
            con.prepareStatement(sql).execute();
        }
        catch ( Exception e ) {
            throw e;
        }
        
        return oldHomeId;       
    }
    

    private int getOldHomeID(String uname) {
        int oldID = -1;
        
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT home_dir_id FROM " + TABLE_PREFIX + "users WHERE user_name='" + uname + "'");
            if ( rs.next () && rs.getInt("home_dir_id") > 0 ) oldID = rs.getInt("home_dir_id");
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "home_error", uname );
        }
        
        return oldID;
    }


    private int getGroupID( String gname ) {
        int id = -1;
    
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "groups WHERE group_name='" + gname + "'");
            if ( rs.next() ) id = rs.getInt("group_id");  
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "group_error", gname );
        }
        
        return id;
    }

    
    private int getUserID ( String uname ) {
        int id = -1;
    
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "users WHERE user_name='" + uname + "'");
            if ( rs.next() ) id = rs.getInt("user_id");  
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            LogManager.getReference().message("error", "user_error", uname );
        }
        
        return id;
    }
    

    private int nextID( String name ) {
        int newID = -1;
             
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT next_id FROM " + TABLE_PREFIX + "sequences WHERE seq_name='" + name + "'");
            if ( rs.next() ) {
                newID = rs.getInt(1);
                con.prepareStatement("UPDATE " + TABLE_PREFIX + "sequences SET next_id=next_id+1 WHERE seq_name='"+name+"'").execute();
            }
            rs.close();stmt.close();
        } 
        catch( Exception e ) {
            LogManager.getReference().message("error", "sequences_error", e.getMessage());
        }
        
        return newID;
    }    


    private void addNewUser(int uID, String uname, String upwd, int hID) throws Exception {
        String sql;
        
        if ( hID != -1 )
            sql = "INSERT INTO " + TABLE_PREFIX + "users VALUES(" + uID + ", '" + uname +"', '" + upwd +"', " + hID + ", 'f')";
        else 
            sql = "INSERT INTO " + TABLE_PREFIX + "users (user_id, user_name, user_passwd, blocked) VALUES(" + uID + ", '" + uname +"', '" + upwd +"', 'f')";
        try {
            con.prepareStatement( sql ).execute();
        }
        catch ( Exception e ) {
            throw e;
        }
    }
    

    private void addUserToGroup( int uID, int gID ) throws Exception {
        try {
            Statement stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM " + TABLE_PREFIX + "ug_relation WHERE user_id=" + uID + " AND group_id=" + gID);
            if ( !rs.next() ) {
                String sql = "INSERT INTO " + TABLE_PREFIX + "ug_relation VALUES (" + uID + ", " + gID + ")";
                con.prepareStatement(sql).execute();
            }
            rs.close();stmt.close();
        }
        catch ( Exception e ) {
            throw e;
        }
    }

     
    private int addNewGroup ( String gname ) throws Exception {
        int gID = -1;
        
        if ( ( gID = getGroupID( gname ) ) == -1 ) {
            if ( ( gID = nextID("group_id") ) == -1 ) throw new Exception();
            String sql = "INSERT INTO " + TABLE_PREFIX + "groups VALUES (" + gID + ", '" + gname + "')";
            try {
                con.prepareStatement(sql).execute();
            }
            catch (Exception e) {
                throw e;
            }
        }
        
        return gID;
    }


    private boolean connect() {
        boolean connected = false;
        try {
            if ( con == null ) con = rdbInfo.getConnection();
            connected = true;
        }
        catch ( Exception e ) {
            errorMsg = LogManager.getReference().getMessage("connection_failed");
            LogManager.getReference().message("error", "statement_failed", e.getMessage());
        }
        return connected;
    }
    

    private void disconnect() {
    }

    
    private class LongFormat {
        String rights;
        String userName;
        String groupName;
        String timeStamp;
        String homeDir;

        
        LongFormat() {
            rights    = null;
            userName  = null;
            groupName = null;
            timeStamp = null;
            homeDir   = null; 
        }

        
        void setRights( String str ) {
            rights = str;
        }

        
        void setUserName( String name ) {
            userName = name;
            setUserSize( name.length() );
        }

                
        void setGroupName( String name ) {
            groupName = name;
            setGroupSize( name.length() );
        }


        void setTimeStamp(String ts) {
            if ( ts.indexOf(".") != -1 ) ts = ts.substring( 0, ts.indexOf(".") );
            timeStamp = ts;
        }

        
        void setHomeDir( String dir ) {
            homeDir = dir;
        }
        

        public String toString() {
            return rights      + "  " + 
                getUserName()  + "  " +
                getGroupName() + "  " +
                timeStamp      + "  " + homeDir;
        }

        
        public String getHomeDir() {
            return homeDir;
        }
        

        private String fixLength(String str, int length) {
            StringBuffer result = new StringBuffer(); 
            for ( int i = str.length(); i < length; i++ ) result.append(" ");
            result.append(str);
            return result.toString();
        }


        private String getUserName() {
            return fixLength(userName, MAX_USER_SIZE);
        }


        private String getGroupName() {
            return fixLength(groupName, MAX_GROUP_SIZE);
        }


        private void setUserSize( int size ) {
            if ( size > MAX_USER_SIZE ) MAX_USER_SIZE = size;
        }


        private void setGroupSize( int size ) {
            if ( size > MAX_GROUP_SIZE ) MAX_GROUP_SIZE = size;
        }
    }
}
